package com.elitescloud.boot.support.serializer;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.annotation.BigDecimalFormat;
import com.elitescloud.boot.provider.DataFormatterProvider;
import com.elitescloud.boot.util.ArrayUtil;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import org.springframework.util.ReflectionUtils;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;

/**
 * BigDecimal类型数据序列号设置.
 *
 * @author Kaiser（wang shao）
 * @date 2024/1/17
 */
public class BigDecimalSerializer extends JsonSerializer<BigDecimal> {

    private final DataFormatterProvider formatterProvider;
    private final Map<Class<?>, Map<String, BigDecimalFormat>> cacheMap = new HashMap<>();

    public BigDecimalSerializer(DataFormatterProvider formatterProvider) {
        this.formatterProvider = formatterProvider;
    }

    @Override
    public void serialize(BigDecimal value, JsonGenerator jsonGenerator, SerializerProvider serializerProvider) throws IOException {
        if (value == null) {
            jsonGenerator.writeNumber((BigDecimal) null);
            return;
        }

        // 格式化值
        var formattedValue = this.formatValue(value, jsonGenerator);

        jsonGenerator.writeNumber(formattedValue);
    }

    private BigDecimal formatValue(BigDecimal value, JsonGenerator jsonGenerator) {
        if (!formatterProvider.supportBigDecimal()) {
            // 未启用
            return value;
        }

        // 获取注解
        var bigDecimalFormat = this.obtainBigDecimalFormat(jsonGenerator);

        // 格式化
        return bigDecimalFormat == null ? value : formatterProvider.format(value, bigDecimalFormat);
    }

    private BigDecimalFormat obtainBigDecimalFormat(JsonGenerator jsonGenerator) {
        // 获取所属类
        var type = jsonGenerator.getCurrentValue();
        if (type == null) {
            return null;
        }

        // 获取字段名
        var context = jsonGenerator.getOutputContext();
        if (context == null) {
            return null;
        }
        var fieldName = context.getCurrentName();
        if (CharSequenceUtil.isBlank(fieldName)) {
            return null;
        }

        // 获取注解
        var typeClass = type.getClass();
        var fieldMap = cacheMap.computeIfAbsent(typeClass, c -> new HashMap<>(64));
        if (fieldMap.containsKey(fieldName)) {
            return fieldMap.get(fieldName);
        }

        var field = ReflectionUtils.findField(typeClass, fieldName);
        if (field == null) {
            return null;
        }
        var annotations = field.getAnnotationsByType(BigDecimalFormat.class);
        var annotation = ArrayUtil.isEmpty(annotations) ? null : annotations[0];
        fieldMap.put(fieldName, annotation);

        return annotation;
    }
}
