package com.elitescloud.boot.util;

import com.elitescloud.boot.SpringContextHolder;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.aop.support.AopUtils;
import org.springframework.cache.interceptor.CacheAspectSupport;
import org.springframework.context.expression.AnnotatedElementKey;
import org.springframework.context.expression.CachedExpressionEvaluator;
import org.springframework.context.expression.MethodBasedEvaluationContext;
import org.springframework.core.BridgeMethodResolver;
import org.springframework.core.ParameterNameDiscoverer;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.Expression;
import org.springframework.util.Assert;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * SpEL表达式解析工具.
 *
 * @author Kaiser（wang shao）
 * @date 2024/2/25
 */
public class CloudtSpelExpressionEvaluator extends CachedExpressionEvaluator {

    private static final CloudtSpelExpressionEvaluator INSTANCE = new CloudtSpelExpressionEvaluator();

    private final Map<AnnotatedElementKey, CloudtSpelMetadata> metadataCache = new ConcurrentHashMap<>(256);
    private final Map<ExpressionKey, Expression> keyCache = new ConcurrentHashMap<>(256);

    private CloudtSpelExpressionEvaluator() {
    }

    /**
     * 获取实例对象
     *
     * @return 实例对象
     */
    public static CloudtSpelExpressionEvaluator getInstance() {
        return INSTANCE;
    }

    /**
     * 创建上下文
     *
     * @param target 目标对象
     * @param method 目标方法
     * @param args   参数
     * @return 上下文
     */
    public EvaluationContext createEvaluationContext(@NotNull Object target, @NotNull Method method, Object[] args) {
        var targetClass = this.getTargetClass(target);
        CloudtSpelMetadata metadata = this.getCloudtSpelMetadata(method, targetClass);
        CloudtSpelRootObject rootObject = new CloudtSpelRootObject(method, args, target, targetClass);

        return new CloudtSpelEvaluationContext(rootObject,
                metadata.getMethod(), args, target, metadata.getTargetClass(), metadata.getTargetMethod(),
                metadata.getMethodKey(), getParameterNameDiscoverer());
    }

    /**
     * 解析表达式
     * <p>
     * 解析表达式时expression可取：
     * <li><b>方法参数：</b>可通过索引访问（从0开始）,如#p0、#a0、#root.args[0]；也可以通过方法名称直接访问，如#orderNo，但不建议这种方式。</li>
     * <li><b>方法名称：</b>#root.methodName</li>
     * <li><b>类的class：</b>#root.targetClass</li>
     *
     * @param evalContext 上下文
     * @param expression  表达式
     * @return 解析结果
     */
    public Object parseExpression(@NotBlank String expression, @NotNull EvaluationContext evalContext) {
        Assert.notNull(expression, "表达式为空");
        Assert.isTrue(evalContext instanceof CloudtSpelEvaluationContext, "暂不支持的evalContext对象");

        CloudtSpelEvaluationContext context = (CloudtSpelEvaluationContext) evalContext;
        return this.getExpression(this.keyCache, context.getMethodKey(), expression).getValue(evalContext);
    }

    /**
     * 直接解析表达式
     *
     * @param expression 表达式
     * @param joinPoint  AOP中的拦截器参数
     * @return 解析结果
     */
    public Object parseExpression(@NotBlank String expression, @NotNull ProceedingJoinPoint joinPoint) {
        Assert.notNull(expression, "表达式为空");
        Assert.notNull(joinPoint, "处理方法为空");

        var context = this.createEvaluationContext(joinPoint.getTarget(), ((MethodSignature) joinPoint.getSignature()).getMethod(), joinPoint.getArgs());
        return this.parseExpression(expression, context);
    }

    private CloudtSpelMetadata getCloudtSpelMetadata(Method method, Class<?> targetClass) {
        // 查看本地缓存是否已有元数据
        AnnotatedElementKey methodKey = new AnnotatedElementKey(method, targetClass);
        var metadata = metadataCache.get(methodKey);

        if (metadata == null) {
            // 初始化元数据
            metadata = new CloudtSpelMetadata(method, targetClass);
            metadataCache.put(methodKey, metadata);
        }

        return metadata;
    }

    private Class<?> getTargetClass(Object target) {
        return AopProxyUtils.ultimateTargetClass(target);
    }

    /**
     * {@link CacheAspectSupport.CacheOperationMetadata}
     */
    static class CloudtSpelMetadata {
        private final Method method;
        private final Class<?> targetClass;
        private final Method targetMethod;
        private final AnnotatedElementKey methodKey;

        public CloudtSpelMetadata(Method method, Class<?> targetClass) {
            this.method = BridgeMethodResolver.findBridgedMethod(method);
            this.targetClass = targetClass;
            this.targetMethod = (!Proxy.isProxyClass(targetClass) ?
                    AopUtils.getMostSpecificMethod(method, targetClass) : this.method);
            this.methodKey = new AnnotatedElementKey(this.targetMethod, targetClass);
        }

        public Method getMethod() {
            return method;
        }

        public Class<?> getTargetClass() {
            return targetClass;
        }

        public Method getTargetMethod() {
            return targetMethod;
        }

        public AnnotatedElementKey getMethodKey() {
            return methodKey;
        }
    }

    /**
     * {@link org.springframework.cache.interceptor.CacheExpressionRootObject}
     */
    static class CloudtSpelRootObject {
        private final Method method;
        private final Object[] args;
        private final Object target;
        private final Class<?> targetClass;

        public CloudtSpelRootObject(Method method, Object[] args, Object target, Class<?> targetClass) {
            this.method = method;
            this.args = args;
            this.target = target;
            this.targetClass = targetClass;
        }

        public Method getMethod() {
            return this.method;
        }

        public String getMethodName() {
            return this.method.getName();
        }

        public Object[] getArgs() {
            return this.args;
        }

        public Object getTarget() {
            return this.target;
        }

        public Class<?> getTargetClass() {
            return this.targetClass;
        }

        /**
         * 获取配置
         *
         * @param key 配置key
         * @return 配置值
         */
        public String getEnv(String key) {
            Assert.hasText(key, "key为空");
            return SpringContextHolder.getProperty(key);
        }
    }

    /**
     * {@link org.springframework.cache.interceptor.CacheEvaluationContext}
     */
    static class CloudtSpelEvaluationContext extends MethodBasedEvaluationContext {
        private final AnnotatedElementKey methodKey;

        public CloudtSpelEvaluationContext(Object rootObject, Method method, Object[] args, Object target, Class<?> targetClass, Method targetMethod,
                                           AnnotatedElementKey methodKey, ParameterNameDiscoverer parameterNameDiscoverer) {
            super(rootObject, targetMethod, args, parameterNameDiscoverer);
            this.methodKey = new AnnotatedElementKey(targetMethod, targetClass);
        }

        public AnnotatedElementKey getMethodKey() {
            return methodKey;
        }
    }
}
