package com.elitescloud.boot.provider.impl;

import com.elitescloud.boot.common.annotation.BigDecimalFormat;
import com.elitescloud.boot.context.properties.DataFormatterProperties;
import com.elitescloud.boot.provider.DataFormatterProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * 默认的格式化数据处理器.
 *
 * @author Kaiser（wang shao）
 * @date 2024/1/16
 */
public class DefaultDataFormatterProvider implements DataFormatterProvider {
    private static final Logger logger = LoggerFactory.getLogger(DefaultDataFormatterProvider.class);

    private final DataFormatterProperties properties;

    public DefaultDataFormatterProvider(DataFormatterProperties properties) {
        this.properties = properties;
    }

    @Override
    public boolean supportBigDecimal() {
        return properties.getBigDecimal() != null && Boolean.TRUE.equals(properties.getBigDecimal().getEnabled());
    }

    @Override
    public BigDecimal format(BigDecimal value, BigDecimalFormat format) {
        return this.execute(value, format);
    }

    private BigDecimal execute(BigDecimal value, BigDecimalFormat format) {
        if (!this.supportBigDecimal()) {
            // 未启用
            return value;
        }

        logger.debug("格式化处理BigDecimal数据：{}", value);
        if (format == null) {
            return value;
        }

        DataFormatterProperties.BigDecimalCfg cfg = null;
        switch (format.bdType()) {
            case AMOUNT:
                cfg = properties.getBigDecimal().getAmount();
                break;
            case PRICE:
                cfg = properties.getBigDecimal().getPrice();
                break;
            case QUANTITY:
                cfg = properties.getBigDecimal().getQuantity();
                break;
            case CUSTOM:
                cfg = new DataFormatterProperties.BigDecimalCfg();
                cfg.setDefaultScale(format.scale());
                cfg.setDefaultRoundingMode(format.roundingMode());
                break;
            default:
                logger.error("暂不支持的数据类型：{}", format.bdType());
                return value;
        }

        var roundingMode = cfg.getDefaultRoundingMode() == null ? RoundingMode.valueOf(cfg.getDefaultRoundingModeValue()) : cfg.getDefaultRoundingMode();
        var formattedValue = value.setScale(cfg.getDefaultScale(), roundingMode);
        logger.debug("格式化处理BigDecimal数据：{}，{}", value, formattedValue);
        return formattedValue;
    }
}
