package com.elitescloud.boot.context;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.spi.TenantContextSpi;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.List;
import java.util.ServiceLoader;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

/**
 * 租户上下文信息工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/26
 */
@Slf4j
public class TenantContextHolder {

    /**
     * 是否启用了租户
     */
    private static boolean enabledTenant;

    private static final List<TenantContextSpi> SPI_LIST = new ArrayList<>();

    static {
        CompletableFuture.runAsync(TenantContextHolder::loadSpiInstances)
                .whenComplete((r, e) -> {
                    if (e == null) {
                        log.info("loaded {} TenantContextSpi instances successfully", SPI_LIST.size());
                        return;
                    }
                    log.error("加载TenantContextSpi失败：", e);
                });
    }

    private TenantContextHolder() {
    }

    /**
     * 租户上下文
     */
    private static final ThreadLocal<SysTenantDTO> CURRENT_TENANT = new ThreadLocal<>();

    /**
     * 获取当前租户信息
     *
     * @return 租户信息
     */
    public static SysTenantDTO getCurrentTenant() {
        return CURRENT_TENANT.get();
    }

    /**
     * 设置当前租户信息
     *
     * @param tenant 租户信息
     */
    public static void setCurrentTenant(SysTenantDTO tenant) {
        Assert.notNull(tenant, "当前租户信息为空");
        CURRENT_TENANT.set(tenant);

        // 调用SPI
        invokeSpiInstance(spi -> spi.onSetTenant(tenant));
    }

    /**
     * 清除当前租户信息
     */
    public static void clearCurrentTenant() {
        var tenant = CURRENT_TENANT.get();
        if (tenant == null) {
            return;
        }
        CURRENT_TENANT.remove();

        // 调用SPI
        invokeSpiInstance(spi -> spi.onClearTenant(tenant));
    }

    /**
     * 是否启用了租户
     *
     * @return 是否启用了租户
     */
    public static boolean isEnabledTenant() {
        return enabledTenant;
    }

    /**
     * 设置是否启用租户
     *
     * @param enabledTenant 启用租户
     */
    static void setEnabledTenant(boolean enabledTenant) {
        TenantContextHolder.enabledTenant = enabledTenant;
    }

    /**
     * 执行SPI异常
     *
     * @param supplier 回调
     */
    private static void invokeSpiInstance(Consumer<TenantContextSpi> supplier) {
        for (TenantContextSpi tenantContextSpi : SPI_LIST) {
            try {
                supplier.accept(tenantContextSpi);
            } catch (Throwable e) {
                log.error("执行租户上下文设置回调异常：", e);
            }
        }
    }

    private static void loadSpiInstances() {
        var spiInstances = ServiceLoader.load(TenantContextSpi.class);
        if (CollUtil.isEmpty(spiInstances)) {
            return;
        }

        for (TenantContextSpi spiInstance : spiInstances) {
            SPI_LIST.add(spiInstance);
        }
    }
}
