package com.elitescloud.boot.util;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.util.function.Supplier;

/**
 * JSON工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/6/1
 */
public class JSONUtil {
    private static final Logger LOG = LoggerFactory.getLogger(JSONUtil.class);
    private static final ObjectMapper OBJECT_MAPPER = ObjectMapperFactory.instance();

    /**
     * 对象转JSON字符串
     *
     * @param obj 对象
     * @return 字符串
     */
    public static String toJsonString(Object obj) {
        return toJsonString(obj, false, null);
    }

    /**
     * 对象转JSON字符串
     *
     * @param obj    对象
     * @param safely 安全转换
     * @return 字符串
     */
    public static String toJsonString(Object obj, boolean safely) {
        return toJsonString(obj, safely, null);
    }

    /**
     * 对象转JSON字符串
     *
     * @param obj      对象
     * @param safely   安全转换
     * @param errorMsg 错误信息
     * @return 字符串
     */
    public static String toJsonString(Object obj, boolean safely, Supplier<String> errorMsg) {
        if (obj == null) {
            return null;
        }

        if (obj.getClass().isPrimitive()) {
            return obj.toString();
        }

        try {
            return OBJECT_MAPPER.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            String eMsg = errorMsg == null ? "转换JSON字符串异常" : errorMsg.get();
            if (safely) {
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.error(eMsg, e);
            return null;
        }
    }

    /**
     * 对象转bytes
     *
     * @param obj 对象
     * @return byte数组
     */
    public static byte[] toJsonBytes(Object obj) {
        return toJsonBytes(obj, false, null);
    }

    /**
     * 对象转bytes
     *
     * @param obj    对象
     * @param safely 安全转换
     * @return byte数组
     */
    public static byte[] toJsonBytes(Object obj, boolean safely) {
        return toJsonBytes(obj, safely, null);
    }

    /**
     * 对象转bytes
     *
     * @param obj      对象
     * @param safely   安全转换
     * @param errorMsg 错误信息
     * @return byte数组
     */
    public static byte[] toJsonBytes(Object obj, boolean safely, Supplier<String> errorMsg) {
        if (obj == null) {
            return new byte[0];
        }

        try {
            return OBJECT_MAPPER.writeValueAsBytes(obj);
        } catch (JsonProcessingException e) {
            String eMsg = errorMsg == null ? "转换byte异常" : errorMsg.get();
            if (safely) {
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.error(eMsg, e);
            return null;
        }
    }

    /**
     * json字符串转对象
     *
     * @param json  json字符串
     * @param clazz 对象类型class
     * @param <T>   对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, Class<T> clazz) {
        return json2Obj(json, clazz, false, null);
    }

    /**
     * json字符串转对象
     *
     * @param json   json字符串
     * @param clazz  对象类型class
     * @param safely 安全转换
     * @param <T>    对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, Class<T> clazz, boolean safely) {
        return json2Obj(json, clazz, safely, null);
    }

    /**
     * json字符串转对象
     *
     * @param json     json字符串
     * @param clazz    对象类型class
     * @param safely   安全转换
     * @param errorMsg 错误信息
     * @param <T>      对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, Class<T> clazz, boolean safely, Supplier<String> errorMsg) {
        if (!StringUtils.hasText(json)) {
            return null;
        }

        try {
            return OBJECT_MAPPER.readValue(json, clazz);
        } catch (JsonProcessingException e) {
            if (safely) {
                String eMsg = errorMsg == null ? "JSON字符串转对象异常" : errorMsg.get();
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.error("json字符串【{}】转对象异常:", json, e);
            return null;
        }
    }

    /**
     * byte转对象
     *
     * @param json  byte数组
     * @param clazz 对象类型class
     * @param <T>   对象类型
     * @return 对象
     */
    public static <T> T jsonBytes2Obj(byte[] json, Class<T> clazz) {
        return jsonBytes2Obj(json, clazz, false, null);
    }

    /**
     * byte转对象
     *
     * @param json   byte数组
     * @param clazz  对象类型class
     * @param safely 安全转换
     * @param <T>    对象类型
     * @return 对象
     */
    public static <T> T jsonBytes2Obj(byte[] json, Class<T> clazz, boolean safely) {
        return jsonBytes2Obj(json, clazz, safely, null);
    }

    /**
     * byte转对象
     *
     * @param json     byte数组
     * @param clazz    对象类型class
     * @param safely   安全转换
     * @param errorMsg 错误信息
     * @param <T>      对象类型
     * @return 对象
     */
    public static <T> T jsonBytes2Obj(byte[] json, Class<T> clazz, boolean safely, Supplier<String> errorMsg) {
        if (ArrayUtil.isEmpty(json)) {
            return null;
        }

        try {
            return OBJECT_MAPPER.readValue(json, clazz);
        } catch (Exception e) {
            if (safely) {
                String eMsg = errorMsg == null ? "byte转对象异常" : errorMsg.get();
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.error("byte转对象异常:", e);
            return null;
        }
    }

    /**
     * json字符串转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json json字符串
     * @param <T>  对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json) {
        return json2Obj(json, false);
    }


    /**
     * json字符串转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json json字符串
     * @param <T>  对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, TypeReference<T> typeReference) {
        return json2Obj(json, typeReference, false, null);
    }

    /**
     * json字符串转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json   json字符串
     * @param safely 安全转换
     * @param <T>    对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, boolean safely) {
        return json2Obj(json, safely, null);
    }

    /**
     * json字符串转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json     json字符串
     * @param safely   安全转换
     * @param errorMsg 错误信息
     * @param <T>      对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, boolean safely, Supplier<String> errorMsg) {
        if (!StringUtils.hasText(json)) {
            return null;
        }

        try {
            return OBJECT_MAPPER.readValue(json, new TypeReference<>() {
            });
        } catch (JsonProcessingException e) {
            if (safely) {
                String eMsg = errorMsg == null ? "JSON字符串转对象异常" : errorMsg.get();
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.error("json字符串【{}】转对象异常:", json, e);
            return null;
        }
    }

    /**
     * byte转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json     json字符串
     * @param safely   安全转换
     * @param errorMsg 错误信息
     * @param <T>      对象类型
     * @return 对象
     */
    public static <T> T json2Obj(String json, TypeReference<T> typeReference, boolean safely, Supplier<String> errorMsg) {
        if (!StringUtils.hasText(json)) {
            return null;
        }

        if (typeReference == null) {
            return json2Obj(json, safely, errorMsg);
        }

        try {
            return OBJECT_MAPPER.readValue(json, typeReference);
        } catch (JsonProcessingException e) {
            if (safely) {
                String eMsg = errorMsg == null ? "JSON字符串转对象异常" : errorMsg.get();
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.error("json字符串【{}】转对象异常:", json, e);
            return null;
        }
    }

    /**
     * byte转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json byte
     * @param <T>  对象类型
     * @return 对象
     */
    public static <T> T jsonBytes2Obj(byte[] json) {
        return jsonBytes2Obj(json, false);
    }


    /**
     * byte转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json byte
     * @param <T>  对象类型
     * @return 对象
     */
    public static <T> T jsonBytes2Obj(byte[] json, TypeReference<T> typeReference) {
        return jsonBytes2Obj(json, typeReference, false, null);
    }

    /**
     * byte转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json   byte
     * @param safely 安全转换
     * @param <T>    对象类型
     * @return 对象
     */
    public static <T> T jsonBytes2Obj(byte[] json, boolean safely) {
        return jsonBytes2Obj(json, safely, null);
    }

    /**
     * byte转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json     byte
     * @param safely   安全转换
     * @param errorMsg 错误信息
     * @param <T>      对象类型
     * @return 对象
     */
    public static <T> T jsonBytes2Obj(byte[] json, boolean safely, Supplier<String> errorMsg) {
        if (ArrayUtil.isEmpty(json)) {
            return null;
        }

        try {
            return OBJECT_MAPPER.readValue(json, new TypeReference<>() {
            });
        } catch (Exception e) {
            if (safely) {
                String eMsg = errorMsg == null ? "JSON字符串转对象异常" : errorMsg.get();
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.error("json字符转对象异常:", e);
            return null;
        }
    }

    /**
     * byte转对象
     * <p>
     * 支持复杂类型的对象转换
     *
     * @param json     bytes
     * @param safely   安全转换
     * @param errorMsg 错误信息
     * @param <T>      对象类型
     * @return 对象
     */
    public static <T> T jsonBytes2Obj(byte[] json, TypeReference<T> typeReference, boolean safely, Supplier<String> errorMsg) {
        if (ArrayUtil.isEmpty(json)) {
            return null;
        }

        if (typeReference == null) {
            return jsonBytes2Obj(json, safely, errorMsg);
        }

        try {
            return OBJECT_MAPPER.readValue(json, typeReference);
        } catch (Exception e) {
            if (safely) {
                String eMsg = errorMsg == null ? "byte转对象异常" : errorMsg.get();
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.error("byte转对象异常:", e);
            return null;
        }
    }

    /**
     * 对象转对象
     *
     * @param object 待转换的对象
     * @param clazz  目标对象的class
     * @param <T>    目标对象类型
     * @return 目标对象
     */
    public static <T> T convertObj(Object object, @NotNull Class<T> clazz) {
        return convertObj(object, clazz, false, null);
    }

    /**
     * 对象转对象
     *
     * @param object 待转换的对象
     * @param clazz  目标对象的class
     * @param safely 是否安全转换
     * @param <T>    目标对象类型
     * @return 目标对象
     */
    public static <T> T convertObj(Object object, @NotNull Class<T> clazz, boolean safely) {
        return convertObj(object, clazz, safely, null);
    }

    /**
     * 对象转对象
     *
     * @param object   待转换的对象
     * @param clazz    目标对象的class
     * @param safely   是否安全转换
     * @param errorMsg 转换失败后的错误信息
     * @param <T>      目标对象类型
     * @return 目标对象
     */
    public static <T> T convertObj(Object object, @NotNull Class<T> clazz, boolean safely, Supplier<String> errorMsg) {
        if (object == null) {
            return null;
        }

        try {
            return OBJECT_MAPPER.convertValue(object, clazz);
        } catch (Exception e) {
            if (safely) {
                String eMsg = errorMsg == null ? "转换对象异常" : errorMsg.get();
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.warn("转对象异常:{}", e.getMessage());
            return null;
        }
    }

    /**
     * 对象转对象
     *
     * @param object        待转换的对象
     * @param typeReference 目标对象的类型
     * @param <T>           目标对象类型
     * @return 目标对象
     */
    public static <T> T convertObj(Object object, @NotNull TypeReference<T> typeReference) {
        return convertObj(object, typeReference, false, null);
    }

    /**
     * 对象转对象
     *
     * @param object        待转换的对象
     * @param typeReference 目标对象的类型
     * @param safely        是否安全转换
     * @param <T>           目标对象类型
     * @return 目标对象
     */
    public static <T> T convertObj(Object object, @NotNull TypeReference<T> typeReference, boolean safely) {
        return convertObj(object, typeReference, safely, null);
    }

    /**
     * 对象转对象
     *
     * @param object        待转换的对象
     * @param typeReference 目标对象的类型
     * @param safely        是否安全转换
     * @param errorMsg      转换失败后的错误信息
     * @param <T>           目标对象类型
     * @return 目标对象
     */
    public static <T> T convertObj(Object object, @NotNull TypeReference<T> typeReference, boolean safely, Supplier<String> errorMsg) {
        if (object == null) {
            return null;
        }

        try {
            return OBJECT_MAPPER.convertValue(object, typeReference);
        } catch (Exception e) {
            if (safely) {
                String eMsg = errorMsg == null ? "转换对象异常" : errorMsg.get();
                throw new IllegalArgumentException(eMsg, e);
            }
            LOG.warn("转对象异常:{}", e.getMessage());
            return null;
        }
    }
}
