package com.elitescloud.boot;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.support.LogStartedInfo;
import com.elitescloud.boot.support.app.CloudtAppHolder;
import lombok.extern.log4j.Log4j2;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.SpringApplicationRunListener;
import org.springframework.boot.autoconfigure.web.ServerProperties;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.io.Resource;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.Field;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 应用生命周期监听器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/14
 */
@Log4j2
class CloudtApplicationRunListener implements SpringApplicationRunListener {

    private final SpringApplication springApplication;

    /**
     * 是否已启动完成
     */
    private static final AtomicBoolean READY = new AtomicBoolean(false);

    public CloudtApplicationRunListener(SpringApplication springApplication, String[] args) {
        this.springApplication = springApplication;
    }

    @Override
    public void ready(ConfigurableApplicationContext context, Duration timeTaken) {
        logStartedInfo(timeTaken);
    }

    private void logStartedInfo(Duration timeTaken) {
        if (!READY.get() && supportLogStartInfo()) {
            if (READY.compareAndSet(false, true)) {
                log.info("系统启动完成，耗时：{}s！", timeTaken.toSeconds());

                String uriPrefix = this.obtainUriPrefix();

                // swagger接口
                String restFulApiUri = generateJavaDocUri(uriPrefix);
                if (StringUtils.hasText(restFulApiUri)) {
                    log.info("可访问RestFul API：{}", restFulApiUri);
                }

                // 其它打印
                var printers = SpringContextHolder.getObjectProvider(LogStartedInfo.class);
                for (LogStartedInfo printer : printers) {
                    printer.print(uriPrefix);
                }
            }
        }
    }

    private String generateJavaDocUri(String uriPrefix) {
        String uri = javaDocUriForSwagger();
        if (StringUtils.hasText(uri)) {
            return uriPrefix + uri;
        }

        uri = javaDocUriForSmartDoc();
        if (StringUtils.hasText(uri)) {
            return uriPrefix + uri;
        }

        return null;
    }

    /**
     * uri前缀
     * <p>
     * protocol + ip + port + contextPath
     *
     * @return
     */
    private String obtainUriPrefix() {
        // 取本机地址
        String contextPath = obtainContextPath();

        String prefix = uriPrefixForDubbo();
        if (StringUtils.hasText(prefix)) {
            return prefix + contextPath;
        }

        String ip = CloudtAppHolder.getServerIp();
        Integer port = SpringContextHolder.getProperty("server.port", Integer.class, 8080);

        return "http://" + ip + ":" + port + contextPath;
    }

    private String obtainContextPath() {
        String contextPath = SpringContextHolder.getProperty("server.servlet.context-path");
        if (!StringUtils.hasText(contextPath)) {
            return "";
        }

        if (!contextPath.startsWith("/")) {
            contextPath = "/" + contextPath;
        }

        if (contextPath.endsWith("/")) {
            contextPath = contextPath.substring(0, contextPath.length() - 1);
        }

        return contextPath;
    }

    private String uriPrefixForDubbo() {
        String ip = SpringContextHolder.getProperty("DUBBO_IP_TO_REGISTRY");
        if (!StringUtils.hasText(ip)) {
            return null;
        }

        Integer port = SpringContextHolder.getProperty("DUBBO_PORT_TO_REGISTRY", Integer.class, 20880);
        String applicationName = SpringContextHolder.getApplicationName();

        return "http://" + ip + ":" + port + "/" + applicationName;
    }

    private String javaDocUriForSwagger() {
        String uri = "/doc.html";

        boolean enabled = SpringContextHolder.getProperty("knife4j.enable", Boolean.class, false);
        if (enabled) {
            return uri;
        }
        return null;
    }

    private String javaDocUriForSmartDoc() {
        Resource index = SpringContextHolder.getResource("classpath:static/doc/index.html");
        if (index != null && index.exists()) {
            return "/doc/index.html";
        }

        return null;
    }

    private boolean supportLogStartInfo() {
        try {
            String fieldName = "logStartupInfo";
            Field field = ReflectionUtils.findField(SpringApplication.class, fieldName);
            if (field == null) {
                log.error("not found field 'logStartupInfo' in SpringApplication !");
                return false;
            }

            boolean canAccess = field.canAccess(springApplication);
            if (!canAccess) {
                field.setAccessible(true);
                boolean result = field.getBoolean(springApplication);
                field.setAccessible(false);
                return result;
            }
            return field.getBoolean(springApplication);
        } catch (IllegalAccessException e) {
            log.error("obtain logStartupInfo fail：", e);
        }

        return false;
    }

    private int getPort(ServerProperties serverProperties) {
        return Objects.requireNonNullElse(serverProperties.getPort(), 8080);
    }

    private String getContextPath(ServerProperties serverProperties) {
        var contextPath = serverProperties.getServlet().getContextPath();
        if (!StringUtils.hasText(contextPath)) {
            return "";
        }

        return "/" + CharSequenceUtil.strip(contextPath, "/");
    }
}
