package com.elitescloud.boot.util;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ClassUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.common.CloudtOptional;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * 对象工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/28
 */
public class ObjUtil extends ObjectUtil {
    private ObjUtil() {
    }

    /**
     * 判断对象是否为空
     *
     * @param obj 对象
     * @return 是否为空
     */
    public static boolean isEmpty(Object obj) {
        if (obj == null) {
            return true;
        }
        if (obj instanceof ApiResult<?>) {
            return isEmpty(((ApiResult<?>)obj).getData());
        }
        if (obj instanceof PagingVO<?>) {
            return ((PagingVO<?>) obj).isEmpty();
        }

        return ObjectUtil.isEmpty(obj);
    }

    /**
     * 如果值不为空时消费
     *
     * @param value    值
     * @param consumer 消费者
     */
    public static <T> void ifNotNull(T value, @NotNull Consumer<T> consumer) {
        if (consumer == null) {
            return;
        }
        if (value != null) {
            consumer.accept(value);
        }
    }

    /**
     * 如果值为空时消费默认值
     *
     * @param value        值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifNull(T value, T defaultValue, @NotNull Consumer<T> consumer) {
        ifNull(value, (Supplier<T>) (() -> defaultValue), consumer);
    }

    /**
     * 如果值为空时消费默认值
     *
     * @param value        值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifNull(T value, Supplier<T> defaultValue, @NotNull Consumer<T> consumer) {
        if (consumer == null) {
            return;
        }
        if (value == null) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 如果值不为空则消费值
     *
     * @param value    待消费的值
     * @param consumer 消费者
     */
    public static void ifNotBlank(String value, @NotNull Consumer<String> consumer) {
        if (consumer == null) {
            return;
        }
        if (StringUtils.hasText(value)) {
            consumer.accept(value);
        }
    }

    /**
     * 如果值为空则消费默认值
     *
     * @param value        待消费的值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static void ifBlank(String value, String defaultValue, @NotNull Consumer<String> consumer) {
        ifBlank(value, () -> defaultValue, consumer);
    }

    /**
     * 如果值为空则消费默认值
     *
     * @param value        待消费的值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static void ifBlank(String value, Supplier<String> defaultValue, @NotNull Consumer<String> consumer) {
        if (consumer == null) {
            return;
        }
        if (!StringUtils.hasText(value)) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 如果集合不为空时消费
     *
     * @param value    集合
     * @param consumer 消费者
     */
    public static <T> void ifNotEmpty(Collection<T> value, @NotNull Consumer<Collection<T>> consumer) {
        if (consumer == null) {
            return;
        }
        if (CollUtil.isNotEmpty(value)) {
            consumer.accept(value);
        }
    }

    /**
     * 如果集合为空时消费默认值
     *
     * @param value        集合
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(Collection<T> value, Collection<T> defaultValue, @NotNull Consumer<Collection<T>> consumer) {
        ifEmpty(value, () -> defaultValue, consumer);
    }

    /**
     * 如果集合为空时消费默认值
     *
     * @param value        集合
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(Collection<T> value, Supplier<Collection<T>> defaultValue, @NotNull Consumer<Collection<T>> consumer) {
        if (consumer == null) {
            return;
        }
        if (CollUtil.isEmpty(value)) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 如果数组不为空时消费
     *
     * @param value    数组
     * @param consumer 消费者
     */
    public static <T> void ifNotEmpty(T[] value, @NotNull Consumer<T[]> consumer) {
        if (consumer == null) {
            return;
        }
        if (ArrayUtil.isNotEmpty(value)) {
            consumer.accept(value);
        }
    }

    /**
     * 如果数组为空时消费默认值
     *
     * @param value        数组
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(T[] value, T[] defaultValue, @NotNull Consumer<T[]> consumer) {
        ifEmpty(value, () -> defaultValue, consumer);
    }

    /**
     * 如果集合为空时消费默认值
     *
     * @param value        集合
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(T[] value, Supplier<T[]> defaultValue, @NotNull Consumer<T[]> consumer) {
        if (consumer == null) {
            return;
        }
        if (ArrayUtil.isEmpty(value)) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 拆包消费
     *
     * @param wrappedObj  包装的对象
     * @param objConsumer 消费者
     * @param <T>         消费对象类型
     */
    public static <T> void unwrap(Object wrappedObj, @NotNull Consumer<T> objConsumer) {
        Class<?> clazz = null;
        // 基本类型
        if (wrappedObj == null || ClassUtil.isSimpleValueType(clazz = wrappedObj.getClass())) {
            return;
        }

        if (wrappedObj instanceof ApiResult) {
            unwrap(((ApiResult<?>) wrappedObj).getData(), objConsumer);
            return;
        }

        if (wrappedObj instanceof PagingVO) {
            for (Object record : ((PagingVO<?>) wrappedObj).getRecords()) {
                unwrap(record, objConsumer);
            }
            return;
        }

        if (wrappedObj instanceof Map<?, ?>) {
            for (Object value : ((Map<?, ?>) wrappedObj).values()) {
                unwrap(value, objConsumer);
            }
            return;
        }

        // 数组
        if (clazz.isArray()) {
            for (Object o : ((Object[]) wrappedObj)) {
                unwrap(o, objConsumer);
            }
            return;
        }

        // 集合
        if (wrappedObj instanceof Iterable<?>) {
            for (Object o : ((Iterable<?>) wrappedObj)) {
                unwrap(o, objConsumer);
            }
            return;
        }

        // Optional翻译
        if (wrappedObj instanceof Optional) {
            ((Optional<?>) wrappedObj).ifPresent(o -> unwrap(o, objConsumer));
            return;
        }
        if (wrappedObj instanceof CloudtOptional) {
            ((CloudtOptional<?>) wrappedObj).ifPresent(o -> unwrap(o, objConsumer));
            return;
        }

        objConsumer.accept((T) wrappedObj);
    }
}
