package com.elitescloud.boot.util;

import cn.hutool.core.lang.UUID;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.RandomUtil;
import org.springframework.http.MediaType;

import java.io.File;
import java.io.IOException;

/**
 * 文件工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/24
 */
public class FileUtil extends cn.hutool.core.io.FileUtil {
    private FileUtil() {
    }

    /**
     * 创建历史文件夹
     *
     * @param dirName 文件夹名
     * @return 文件夹
     */
    public static File createTempDir(String... dirName) {
        if (ArrayUtil.isEmpty(dirName)) {
            dirName = new String[]{UUID.fastUUID().toString(true) + RandomUtil.randomString(4)};
        }

        File dir = new File(System.getProperty("java.io.tmpdir"));
        if (!dir.exists()) {
            dir.mkdirs();
        }
        dir = new File(dir, String.join(File.separator, dirName));
        dir.mkdirs();

        return dir;
    }

    /**
     * 创建临时文件
     *
     * @param fileName 文件名
     * @param dirName  目录
     * @return 临时文件
     */
    public static File createTempFile(String fileName, String... dirName) {
        if (CharSequenceUtil.isBlank(fileName)) {
            fileName = UUID.fastUUID().toString(true) + RandomUtil.randomString(4);
        }

        File dir = new File(System.getProperty("java.io.tmpdir"));
        if (ArrayUtil.isNotEmpty(dirName)) {
            if (!dir.exists()) {
                dir.mkdirs();
            }
            dir = new File(dir, String.join(File.separator, dirName));
            dir.mkdirs();
        }
        File file = new File(dir, fileName);
        if (file.exists()) {
            file.delete();
        }
        try {
            file.createNewFile();
        } catch (IOException e) {
            throw new RuntimeException("创建临时文件失败", e);
        }
        return file;
    }

    /**
     * 根据文件名获取mime-type
     *
     * @param filename
     * @return
     */
    public static String convertMimeTypeByFilename(String filename) {
        if (CharSequenceUtil.isBlank(filename)) {
            return null;
        }

        return cn.hutool.core.io.FileUtil.getMimeType(filename);
    }

    /**
     * mime-type转 mediaType
     *
     * @param mimeType 文件类型
     * @return 媒体类型
     */
    public static MediaType parseMediaType(String mimeType) {
        if (CharSequenceUtil.isBlank(mimeType)) {
            return null;
        }
        var types = mimeType.split("/");
        return types.length > 1 ? new MediaType(types[0], types[1]) : new MediaType(types[0]);
    }
}
