package com.elitescloud.boot.util;

import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.exception.BusinessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.net.URL;
import java.util.Properties;

/**
 * CloudtBoot工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/7/24
 */
public class CloudtBootUtil {
    private static final Logger logger = LoggerFactory.getLogger(CloudtBootUtil.class);
    private static final Package PACKAGE = CloudtBootUtil.class.getPackage();

    private CloudtBootUtil() {
    }

    /**
     * 获取架构版本
     *
     * @return 版本号
     */
    public static String getVersion() {
        return PACKAGE.getImplementationVersion();
    }

    /**
     * 获取当前项目的版本号
     *
     * @return 版本号
     */
    public static String getProjectVersion() {
        var mainClass = SpringContextHolder.getMainClass();
        Assert.notNull(mainClass, "服务状态异常，无法获取到启动类");
        var version = mainClass.getPackage().getImplementationVersion();
        if (StringUtils.hasText(version)) {
            return version;
        }

        // 本地启动项目时，从本地构建文件获取
        CloudtProjectInfo projectInfo = null;
        try {
            projectInfo = getBuildInfo();
        } catch (Exception e) {
            logger.error("加载本地构建文件异常：", e);
        }
        return projectInfo == null ? null : projectInfo.getProjectVersion();
    }

    /**
     * 获取本地构建文件信息
     *
     * @return 文件信息
     */
    public static CloudtProjectInfo getBuildInfo() {
        // 根据启动类获取构建文件
        var mainClassUrl = SpringContextHolder.getMainClass().getClassLoader().getResource("");
        if (mainClassUrl == null) {
            return null;
        }
        var urlPath = mainClassUrl.toString();
        var buildIndex = urlPath.indexOf("/build/");
        if (buildIndex < 1) {
            return null;
        }
        urlPath = urlPath.substring(0, buildIndex) + "/build/cloudtProjectInfo.properties";

        Properties properties = new Properties();
        try {
            properties.load(new URL(urlPath).openStream());
        } catch (Exception e) {
            throw new BusinessException("加载配置文件失败", e);
        }
        return new CloudtProjectInfo(properties);
    }

    public static class CloudtProjectInfo {
        private final Properties properties;
        /**
         * 名称
         */
        private String projectName;
        /**
         * 版本
         */
        private String projectVersion;
        /**
         * 坐标-group
         */
        private String projectGroup;
        /**
         * 项目描述
         */
        private String projectDescription;
        /**
         * 项目状态
         */
        private String projectStatus;
        /**
         * 项目构建时间
         */
        private String projectBuildTime;
        /**
         * 用户
         */
        private String userName;
        /**
         * java版本
         */
        private String javaVersion;
        /**
         * 操作系统名称
         */
        private String osName;
        /**
         * 主机地址
         */
        private String osHost;
        /**
         * 主机名称
         */
        private String osHostname;

        public CloudtProjectInfo(Properties properties) {
            this.properties = properties;
        }

        public String getProjectName() {
            return properties.getProperty("project.name");
        }

        public String getProjectVersion() {
            return properties.getProperty("project.version");
        }

        public String getProjectGroup() {
            return properties.getProperty("project.group");
        }

        public String getProjectDescription() {
            return properties.getProperty("project.description");
        }

        public String getProjectStatus() {
            return properties.getProperty("project.status");
        }

        public String getProjectBuildTime() {
            return properties.getProperty("project.buildTime");
        }

        public String getUserName() {
            return properties.getProperty("user.name");
        }

        public String getJavaVersion() {
            return properties.getProperty("java.version");
        }

        public String getOsName() {
            return properties.getProperty("os.name");
        }

        public String getOsHost() {
            return properties.getProperty("os.host");
        }

        public String getOsHostname() {
            return properties.getProperty("os.hostname");
        }
    }
}
