package com.elitescloud.boot.util;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.util.Assert;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.time.Duration;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

/**
 * 锁工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/6
 */
@Slf4j
public class LockUtil {
    private static final ConcurrentHashMap<String, Lock> LOCAL_LOCK = new ConcurrentHashMap<>(512);
    private static RedissonClient redissonClient;

    private LockUtil() {
    }

    /**
     * 加锁执行
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier) {
        return executeByLock(lockKey, supplier, null);
    }

    /**
     * 加锁执行
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param duration 最多等待获取锁的时间
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier, Duration duration) {
        return executeByLock(lockKey, supplier, duration, null);
    }

    /**
     * 加锁执行
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param duration 最多等待获取锁的时间
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier, Duration duration, String acquireFailMsg) {
        Assert.hasText(lockKey, "锁的标识为空");
        if (duration == null) {
            duration = Duration.ofMinutes(10);
        }

        // 优先获取本地锁
        var lockLocal = LOCAL_LOCK.computeIfAbsent(lockKey, c -> new ReentrantLock());
        RLock lockRemote = null;
        try {
            log.info("尝试获取本地锁：{}，{}", lockKey, duration);
            if (lockLocal.tryLock(duration.getSeconds(), TimeUnit.SECONDS)) {
                // 尝试获取远程锁
                lockRemote = getRedissonClient().getLock(lockKey);
                log.info("尝试获取远程锁：{}，{}", lockKey, duration);
                if (lockRemote.tryLock(duration.getSeconds(), TimeUnit.SECONDS)) {
                    return supplier.get();
                }
            }

            throw new BusinessException(CharSequenceUtil.blankToDefault(acquireFailMsg, "未获取到锁" + lockKey + "，等待超时"));
        } catch (Throwable e) {
            log.error("加锁的方法中出现异常：{}", lockKey, e);
            if (e instanceof RuntimeException) {
                throw (BusinessException) e;
            }
            throw new RuntimeException(e);
        } finally {
            // 释放锁
            log.info("释放本地锁：{}", lockKey);
            try {
                lockLocal.unlock();
            } catch (Exception e) {
                log.error("本地锁释放异常：{}", lockKey, e);
            }
            if (lockRemote != null && lockRemote.isLocked()) {
                log.info("释放远程锁：{}", lockKey);
                try {
                    lockRemote.unlock();
                } catch (Exception e) {
                    log.error("远程锁释放异常：{}", lockKey, e);
                }
            }
        }
    }

    private static RedissonClient getRedissonClient() {
        if (redissonClient == null) {
            redissonClient = SpringContextHolder.getBean(RedissonClient.class);
        }
        return redissonClient;
    }
}
