package com.elitescloud.boot.common.translate.translator;

import com.elitescloud.boot.common.translate.TranslatorCustomizer;
import com.elitescloud.boot.util.BeanWrapperUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.slf4j.Slf4j;

import javax.validation.constraints.NotNull;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/8
 */
@Slf4j
public abstract class AbstractTranslatorCustomizer implements TranslatorCustomizer {

    /**
     * 执行翻译
     *
     * @param data                待翻译的数据
     * @param targetFieldSupplier 获取待翻译的字段
     * @param targetValueSupplier 获取翻译后的值
     */
    protected void executeTranslate(@NotNull Object data, @NotNull Function<BeanWrapperUtil.CustomField, String> targetFieldSupplier,
                                    @NotNull BiFunction<BeanWrapperUtil.CustomField, Object, Object> targetValueSupplier) {
        this.executeTranslate(data, targetFieldSupplier, targetValueSupplier, false);
    }

    /**
     * 执行翻译
     *
     * @param data                待翻译的数据
     * @param targetFieldSupplier 获取待翻译的字段
     * @param targetValueSupplier 获取翻译后的值
     * @param reversal            是否反转字段，true：targetField表示是被翻译的字段，false：targetField表示是翻译后的字段（相当于name字段）
     */
    protected void executeTranslate(@NotNull Object data, @NotNull Function<BeanWrapperUtil.CustomField, String> targetFieldSupplier,
                                    @NotNull BiFunction<BeanWrapperUtil.CustomField, Object, Object> targetValueSupplier, boolean reversal) {
        // 获取所有字段
        var fieldMap = BeanWrapperUtil.getFields(data).stream().collect(Collectors.toMap(t -> t.getField().getName(), Function.identity(), (t1, t2) -> t1));
        if (fieldMap.isEmpty()) {
            return;
        }

        String targetFieldName = null;
        BeanWrapperUtil.CustomField targetField = null;
        DataAccessHelper dataAccessHelper = new DataAccessHelper(data);
        for (Map.Entry<String, BeanWrapperUtil.CustomField> entry : fieldMap.entrySet()) {
            // 获取要翻译的字段
            targetFieldName = targetFieldSupplier.apply(entry.getValue());
            if (targetFieldName == null) {
                continue;
            }
            targetField = fieldMap.get(targetFieldName);
            if (targetField == null) {
                log.warn("翻译忽略，未获取到{}的翻译字段：{}", entry.getValue().toString(), targetFieldName);
                continue;
            }

            // 获取待翻译的值
            Object value = null;
            try {
                value = dataAccessHelper.readProperty(reversal ? targetField : entry.getValue());
            } catch (Exception e) {
                log.warn("翻译忽略，获取{}的值异常：", (reversal ? targetField : entry.getValue()).toString(), e);
                continue;
            }
            if (ObjUtil.isEmpty(value)) {
                continue;
            }

            // 获取翻译后的值
            Object result = targetValueSupplier.apply(entry.getValue(), value);
            if (ObjUtil.isEmpty(result)) {
                continue;
            }

            // 写入翻译后的值
            try {
                dataAccessHelper.setProperty(reversal ? entry.getValue() : targetField, result);
            } catch (Exception e) {
                log.error("翻译忽略，向{}写入翻译结果异常：", reversal ? entry.getValue() : targetField, e);
            }
        }
    }

    protected String cacheKeyPrefix(SysTenantDTO tenant) {
        return tenant == null ? "default_" : "tenant_" + tenant.getTenantCode() + "_";
    }

    protected static class DataAccessHelper {
        private final Object data;

        public DataAccessHelper(Object data) {
            this.data = data;
        }

        /**
         * 获取属性
         *
         * @param customField 字段
         * @return 字段值
         * @throws Exception 异常
         */
        public Object readProperty(@NotNull BeanWrapperUtil.CustomField customField) throws Exception {
            return this.readProperty(customField, false);
        }

        /**
         * 获取属性的值
         *
         * @param customField 字段
         * @param required    是否必须支持
         * @return 字段值
         * @throws Exception 异常
         */
        public Object readProperty(@NotNull BeanWrapperUtil.CustomField customField, boolean required) throws Exception {
            if (customField.getReadMethod() == null) {
                if (!required) {
                    return null;
                }
                throw new IllegalAccessException(customField.getField().getName() + "缺少get方法");
            }

            return customField.getReadMethod().invoke(data);
        }

        /**
         * 设置属性的值
         *
         * @param customField 字段
         * @param value       值
         * @throws Exception 异常
         */
        public void setProperty(@NotNull BeanWrapperUtil.CustomField customField, Object value) throws Exception {
            this.setProperty(customField, value, false);
        }

        /**
         * 设置属性的值
         *
         * @param customField 字段
         * @param value       值
         * @param required    是否必须支持
         * @throws Exception 异常
         */
        public void setProperty(@NotNull BeanWrapperUtil.CustomField customField, Object value, boolean required) throws Exception {
            if (customField.getWriteMethod() == null) {
                if (!required) {
                    return;
                }
                throw new IllegalAccessException(customField.getField().getName() + "缺少set方法");
            }

            customField.getWriteMethod().invoke(data, value);
        }
    }
}
