package com.elitescloud.boot.util;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ClassUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.common.CloudtOptional;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * 对象工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/28
 */
public class ObjUtil extends ObjectUtil {
    private ObjUtil() {
    }

    /**
     * 判断对象是否为空
     *
     * @param obj 对象
     * @return 是否为空
     */
    public static boolean isEmpty(Object obj) {
        if (obj == null) {
            return true;
        }
        if (obj instanceof ApiResult<?>) {
            return isEmpty(((ApiResult<?>) obj).getData());
        }
        if (obj instanceof PagingVO<?>) {
            return ((PagingVO<?>) obj).isEmpty();
        }

        return ObjectUtil.isEmpty(obj);
    }

    /**
     * 如果值不为空时消费
     *
     * @param value    值
     * @param consumer 消费者
     */
    public static <T> void ifNotNull(T value, @NotNull Consumer<T> consumer) {
        if (consumer == null) {
            return;
        }
        if (value != null) {
            consumer.accept(value);
        }
    }

    /**
     * 如果值为空时消费默认值
     *
     * @param value        值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifNull(T value, T defaultValue, @NotNull Consumer<T> consumer) {
        ifNull(value, (Supplier<T>) (() -> defaultValue), consumer);
    }

    /**
     * 如果值为空时消费默认值
     *
     * @param value        值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifNull(T value, Supplier<T> defaultValue, @NotNull Consumer<T> consumer) {
        if (consumer == null) {
            return;
        }
        if (value == null) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 如果值不为空则消费值
     *
     * @param value    待消费的值
     * @param consumer 消费者
     */
    public static void ifNotBlank(String value, @NotNull Consumer<String> consumer) {
        if (consumer == null) {
            return;
        }
        if (StringUtils.hasText(value)) {
            consumer.accept(value);
        }
    }

    /**
     * 如果值为空则消费默认值
     *
     * @param value        待消费的值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static void ifBlank(String value, String defaultValue, @NotNull Consumer<String> consumer) {
        ifBlank(value, () -> defaultValue, consumer);
    }

    /**
     * 如果值为空则消费默认值
     *
     * @param value        待消费的值
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static void ifBlank(String value, Supplier<String> defaultValue, @NotNull Consumer<String> consumer) {
        if (consumer == null) {
            return;
        }
        if (!StringUtils.hasText(value)) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 如果集合不为空时消费
     *
     * @param value    集合
     * @param consumer 消费者
     */
    public static <T> void ifNotEmpty(Collection<T> value, @NotNull Consumer<Collection<T>> consumer) {
        if (consumer == null) {
            return;
        }
        if (CollUtil.isNotEmpty(value)) {
            consumer.accept(value);
        }
    }

    /**
     * 如果集合为空时消费默认值
     *
     * @param value        集合
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(Collection<T> value, Collection<T> defaultValue, @NotNull Consumer<Collection<T>> consumer) {
        ifEmpty(value, () -> defaultValue, consumer);
    }

    /**
     * 如果集合为空时消费默认值
     *
     * @param value        集合
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(Collection<T> value, Supplier<Collection<T>> defaultValue, @NotNull Consumer<Collection<T>> consumer) {
        if (consumer == null) {
            return;
        }
        if (CollUtil.isEmpty(value)) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 如果数组不为空时消费
     *
     * @param value    数组
     * @param consumer 消费者
     */
    public static <T> void ifNotEmpty(T[] value, @NotNull Consumer<T[]> consumer) {
        if (consumer == null) {
            return;
        }
        if (ArrayUtil.isNotEmpty(value)) {
            consumer.accept(value);
        }
    }

    /**
     * 如果数组为空时消费默认值
     *
     * @param value        数组
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(T[] value, T[] defaultValue, @NotNull Consumer<T[]> consumer) {
        ifEmpty(value, () -> defaultValue, consumer);
    }

    /**
     * 如果集合为空时消费默认值
     *
     * @param value        集合
     * @param defaultValue 默认值
     * @param consumer     消费者
     */
    public static <T> void ifEmpty(T[] value, Supplier<T[]> defaultValue, @NotNull Consumer<T[]> consumer) {
        if (consumer == null) {
            return;
        }
        if (ArrayUtil.isEmpty(value)) {
            consumer.accept(defaultValue.get());
        }
    }

    /**
     * 拆包消费
     *
     * @param wrappedObj  包装的对象
     * @param objConsumer 消费者
     * @param <T>         消费对象类型
     */
    public static <T> void unwrap(Object wrappedObj, @NotNull Consumer<T> objConsumer) {
        Class<?> clazz = null;
        // 基本类型
        if (wrappedObj == null || ClassUtil.isSimpleValueType(clazz = wrappedObj.getClass())) {
            return;
        }

        if (wrappedObj instanceof ApiResult) {
            unwrap(((ApiResult<?>) wrappedObj).getData(), objConsumer);
            return;
        }

        if (wrappedObj instanceof PagingVO) {
            for (Object record : ((PagingVO<?>) wrappedObj).getRecords()) {
                unwrap(record, objConsumer);
            }
            return;
        }

        if (wrappedObj instanceof Map<?, ?>) {
            for (Object value : ((Map<?, ?>) wrappedObj).values()) {
                unwrap(value, objConsumer);
            }
            return;
        }

        // 数组
        if (clazz.isArray()) {
            for (Object o : ((Object[]) wrappedObj)) {
                unwrap(o, objConsumer);
            }
            return;
        }

        // 集合
        if (wrappedObj instanceof Iterable<?>) {
            for (Object o : ((Iterable<?>) wrappedObj)) {
                unwrap(o, objConsumer);
            }
            return;
        }

        // Optional翻译
        if (wrappedObj instanceof Optional) {
            ((Optional<?>) wrappedObj).ifPresent(o -> unwrap(o, objConsumer));
            return;
        }
        if (wrappedObj instanceof CloudtOptional) {
            ((CloudtOptional<?>) wrappedObj).ifPresent(o -> unwrap(o, objConsumer));
            return;
        }

        objConsumer.accept((T) wrappedObj);
    }

    /**
     * 转换基本数据类型
     *
     * @param obj      待转换的数据类型
     * @param javaType java类型
     * @return 结果
     */
    public static Object convertBasicValue(Object obj, @NotNull String javaType) {
        if (obj == null) {
            return null;
        }
        switch (javaType) {
            case "java.lang.String":
                return obj.toString();
            case "java.lang.Long":
                return Long.valueOf(obj.toString());
            case "java.lang.Integer":
                return Integer.valueOf(obj.toString());
            case "java.lang.Boolean":
                return Boolean.valueOf(obj.toString());
            case "java.lang.Double":
                return Double.valueOf(obj.toString());
            case "java.lang.Float":
                return Float.valueOf(obj.toString());
            case "java.lang.Short":
                return Short.valueOf(obj.toString());
            case "java.lang.Character":
                return obj.toString().charAt(0);
            case "java.lang.Byte":
                return Byte.valueOf(obj.toString());
            case "java.time.LocalDateTime":
                return DatetimeUtil.parseLocalDateTime(obj.toString());
            case "java.time.LocalDate":
                return DatetimeUtil.parseLocalDate(obj.toString());
            case "java.util.Date":
                return DatetimeUtil.parseForDate(obj.toString());
            default:
                throw new IllegalArgumentException("暂不支持的数据类型：" + javaType);
        }
    }
}
