package com.elitescloud.boot.auth;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.constant.AuthenticationClaim;
import lombok.Getter;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 认证过的客户端.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/6
 */
@Getter
public class AuthorizedClient implements Serializable {
    private static final long serialVersionUID = -2450603428618865616L;

    /**
     * 客户端ID
     */
    private String clientId;

    /**
     * 授权范围
     */
    private List<String> scopes;

    /**
     * 授权租户
     */
    private List<String> tenantCodes;

    @SuppressWarnings("unchecked")
    public static AuthorizedClient buildByJwt(Jwt jwt) {
        AuthorizedClient authorizedClient = new AuthorizedClient();

        authorizedClient.setClientId(obtainClientId(jwt));

        var scopes = jwt.getClaimAsStringList("scope");
        authorizedClient.setScopes(ObjectUtil.defaultIfNull(scopes, Collections.emptyList()));

        var tenantCodeStr = jwt.getClaimAsString(AuthenticationClaim.KEY_AUTHED_TENANTS);
        authorizedClient.setTenantCodes(StringUtils.hasText(tenantCodeStr) ? Arrays.stream(tenantCodeStr.split(",")).collect(Collectors.toList()) : Collections.emptyList());

        return authorizedClient;
    }

    private void setClientId(String clientId) {
        this.clientId = clientId;
    }

    private void setScopes(List<String> scopes) {
        this.scopes = scopes;
    }

    private void setTenantCodes(List<String> tenantCodes) {
        this.tenantCodes = tenantCodes;
    }

    private static String obtainClientId(Jwt jwt) {
        var clientId = jwt.getClaimAsString(AuthenticationClaim.KEY_CLIENT_ID);
        if (StringUtils.hasText(clientId)) {
            return clientId;
        }
        var audience = jwt.getAudience();
        if (!CollectionUtils.isEmpty(audience)) {
            return audience.get(0);
        }
        return null;
    }
}

