package com.elitescloud.boot.model.entity;

import com.elitescloud.boot.constant.IdGeneratorStrategy;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.cloudt.common.annotation.Comment;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;
import org.hibernate.annotations.*;
import org.springframework.data.jpa.domain.support.AuditingEntityListener;

import javax.persistence.*;
import java.io.Serializable;
import java.time.LocalDateTime;
import java.util.Objects;

/**
 * 实体的基础类
 *
 * <pre>
 *     JPA数据实体类的基础类
 *     唯一ID生成策略绑定
 *     审计字段、版本和逻辑删除标记
 *     租户ID、租户组织ID与组织ID
 * </pre>
 *
 * @author Moz
 * 3/17/2020
 */
@Getter
@Setter
@Accessors(chain = true)
@Where(clause = "delete_flag = 0 or delete_flag is null")
@FilterDef(name = TenantConstant.ENTITY_FILTER_NAME, defaultCondition = ":tenant_id = tenant_id",
        parameters = {@ParamDef(name = "tenant_id", type = "long")}
)
@Filter(name = TenantConstant.ENTITY_FILTER_NAME)
@FilterDef(name = TenantConstant.ENTITY_FILTER_ORG_NAME, defaultCondition = ":tenant_org_id = tenant_org_id",
        parameters = {@ParamDef(name = "tenant_org_id", type = "long")}
)
@Filter(name = TenantConstant.ENTITY_FILTER_ORG_NAME)
@EntityListeners(AuditingEntityListener.class)
@MappedSuperclass
public abstract class BaseModel implements Serializable {
    private static final long serialVersionUID = -6696939650444719776L;

    @PreUpdate
    @PrePersist
    public void updateTimestamp() {
        this.modifyTime = LocalDateTime.now();
    }

    @Id
    @GenericGenerator(name = "el-id", strategy = IdGeneratorStrategy.DEFAULT)
    @GeneratedValue(strategy = GenerationType.AUTO, generator = "el-id")
    @Comment("记录唯一ID")
    @Column()
    private Long id;

    @Comment(value = "所属租户ID", defaultValue = "-1")
    @Column(updatable = false)
    private Long tenantId;

    @Comment(value = "所属组织ID")
    @Column(updatable = false)
    private Long belongOrgId;

    @Comment(value = "租户组织ID")
    @Column(updatable = false)
    private Long tenantOrgId;

    @Comment("备注")
    @Column()
    private String remark;

    @Comment("记录创建者ID")
    @Column(updatable = false)
    private Long createUserId;

    @Comment("记录创建者")
    @Column(updatable = false)
    private String creator;

    @Comment("记录创建时间")
    @Column(updatable = false)
    private LocalDateTime createTime;

    @Comment("记录最后更新者ID")
    @Column()
    private Long modifyUserId;

    @Comment("记录最后更新者")
    @Column()
    private String updater;

    @Comment("记录最后更新时间")
    @Column()
    private LocalDateTime modifyTime;

    @Comment(value = "删除标记", defaultValue = "0")
    @Column()
    private Integer deleteFlag = 0;

    @Comment("锁版本")
    @Column()
    private Integer auditDataVersion;

    @Deprecated(since = "3.2", forRemoval = true)
    @Comment("数据归属组织id")
    @Column()
    private Long secBuId;

    @Deprecated(since = "3.2", forRemoval = true)
    @Comment("数据归属雇员id")
    @Column()
    private Long secUserId;

    @Deprecated(since = "3.2", forRemoval = true)
    @Comment("数据归属公司id")
    @Column()
    private Long secOuId;

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o instanceof BaseModel) {
            BaseModel b = (BaseModel) o;
            return Objects.equals(getId(), b.getId());
        }

        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id);
    }
}
