package com.elitescloud.boot.util;

import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ClassUtil;
import cn.hutool.core.util.ReflectUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.PropertyAccessorFactory;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Bean访问工具.
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/8
 */
@Slf4j
public class BeanWrapperUtil {
    private static final Map<Class<?>, TypeWrapper> WRAPPER_MAP = new HashMap<>(512);

    /**
     * 获取数据的属性值
     *
     * @param data       待获取的数据
     * @param fieldNames 属性名
     * @return 树形和值
     */
    public static Map<String, Object> getFieldValue(@NotNull Object data, @NotEmpty String[] fieldNames) {
        Assert.notNull(data, "数据为空");
        Assert.notEmpty(fieldNames, "修改的属性为空");

        var propertyAccessor = PropertyAccessorFactory.forDirectFieldAccess(data);
        Map<String, Object> valueMap = new HashMap<>(fieldNames.length);
        for (String fieldName : fieldNames) {
            valueMap.put(fieldName, propertyAccessor.getPropertyValue(fieldName));
        }
        return valueMap;
    }

    /**
     * 修改数据的属性值
     *
     * @param data        待修改的数据
     * @param fieldValues 属性和值
     */
    public static void updateFieldValue(@NotNull Object data, @NotEmpty Map<String, Object> fieldValues) {
        updateFieldValue(data, fieldValues, false);
    }

    /**
     * 修改数据的属性值
     *
     * @param data            待修改的数据
     * @param fieldValues     属性和值
     * @param ignoreException 是否忽略异常
     */
    public static void updateFieldValue(@NotNull Object data, @NotEmpty Map<String, Object> fieldValues, boolean ignoreException) {
        Assert.notNull(data, "数据为空");
        Assert.notEmpty(fieldValues, "修改的属性值为空");

        var propertyAccessor = PropertyAccessorFactory.forDirectFieldAccess(data);
        for (Map.Entry<String, Object> entry : fieldValues.entrySet()) {
            if (!StringUtils.hasText(entry.getKey()) || !propertyAccessor.isWritableProperty(entry.getKey())) {
                continue;
            }

            try {
                propertyAccessor.setPropertyValue(entry.getKey(), entry.getValue());
            } catch (BeansException e) {
                log.error("修改属性值异常：{}，{}", entry.getKey(), entry.getValue());
                if (!ignoreException) {
                    throw new IllegalArgumentException(e);
                }
            }
        }
    }

    /**
     * 修改数据的属性值
     *
     * @param data               待修改的数据
     * @param fieldNames         属性和值
     * @param fieldValueFunction 属性值，可根据属性类型获取值
     */
    public static void updateFieldValue(@NotNull Object data, @NotEmpty String[] fieldNames, Function<Class<?>, Object> fieldValueFunction) {
        updateFieldValue(data, fieldNames, fieldValueFunction, false);
    }

    /**
     * 修改数据的属性值
     *
     * @param data               待修改的数据
     * @param fieldNames         属性和值
     * @param fieldValueFunction 属性值，可根据属性类型获取值
     * @param ignoreException    是否忽略异常
     */
    public static void updateFieldValue(@NotNull Object data, @NotEmpty String[] fieldNames, Function<Class<?>, Object> fieldValueFunction, boolean ignoreException) {
        Assert.notNull(data, "数据为空");
        Assert.notEmpty(fieldNames, "修改的属性为空");
        Assert.notNull(fieldValueFunction, "获取值的方法为空");

        var propertyAccessor = PropertyAccessorFactory.forDirectFieldAccess(data);
        Class<?> fieldType = null;
        Object value = null;
        for (var fieldName : fieldNames) {
            if (!StringUtils.hasText(fieldName) || !propertyAccessor.isWritableProperty(fieldName)) {
                continue;
            }
            fieldType = propertyAccessor.getPropertyType(fieldName);
            if (fieldType == null) {
                log.warn("属性不存在：{}", fieldName);
                continue;
            }

            value = fieldValueFunction.apply(fieldType);
            try {
                propertyAccessor.setPropertyValue(fieldName, value);
            } catch (BeansException e) {
                log.error("修改属性值异常：{}，{}", fieldName, value);
                if (!ignoreException) {
                    throw new IllegalArgumentException(e);
                }
            }
        }
    }

    /**
     * 获取所有属性
     * <p>
     * 包含继承的父类的属性
     *
     * @param data 数据对象
     * @return 属性列表
     */
    public static List<CustomField> getFields(@NotNull Object data) {
        return getFields(data, true);
    }

    /**
     * 获取属性
     *
     * @param data       数据对象
     * @param withParent 是否包含父类的属性
     * @return 属性列表
     */
    public static List<CustomField> getFields(@NotNull Object data, boolean withParent) {
        Assert.notNull(data, "数据为空");

        return getFields(data, null, withParent);
    }

    /**
     * 获取属性
     *
     * @param data 数据对象
     * @return 属性列表
     */
    public static CustomField getField(@NotNull Object data, @NotBlank String fieldName) {
        Assert.notNull(data, "数据为空");
        Assert.hasText(fieldName, "属性名称为空");

        var field = getTypeWrapper(data).fieldNames.get(fieldName);
        return field == null ? null : field.toCustomField();
    }

    /**
     * 根据注解获取属性
     *
     * @param data       数据对象
     * @param annotation 注解类型
     * @return 属性列表
     */
    public static List<CustomField> getFieldsByAnnotation(@NotNull Object data, @NotNull Class<? extends Annotation> annotation) {
        return getFieldsByAnnotation(data, annotation, true);
    }

    /**
     * 根据注解获取属性
     *
     * @param data       数据对象
     * @param annotation 注解类型
     * @return 属性列表
     */
    public static List<CustomField> getFieldsByAnnotation(@NotNull Object data, @NotNull Class<? extends Annotation> annotation, boolean withParent) {
        Assert.notNull(data, "数据为空");
        Assert.notNull(annotation, "注解类型为空");

        return getFields(data, annotation, withParent);
    }

    /**
     * 获取属性的注解
     * <p>
     * 如果有多个，则返回第一个
     *
     * @param field          属性
     * @param annotationType 注解类型
     * @return 注解
     */
    public static <T extends Annotation> T getAnnotation(@NotNull Field field, @NotNull Class<T> annotationType) {
        var annotations = getAnnotations(field, annotationType);
        return annotations.isEmpty() ? null : annotations.get(0);
    }

    /**
     * 获取属性的注解
     * <p>
     * 如果有多个，则返回第一个
     *
     * @param data           数据
     * @param fieldName      属性名称
     * @param annotationType 注解类型
     * @return 注解
     */
    public static <T extends Annotation> T getAnnotation(@NotNull Object data, @NotBlank String fieldName, @NotNull Class<T> annotationType) {
        var annotations = getAnnotations(data, fieldName, annotationType);
        return annotations.isEmpty() ? null : annotations.get(0);
    }

    /**
     * 获取属性的注解
     * <p>
     * 如果有多个，则返回第一个
     *
     * @param field          属性
     * @param annotationType 注解类型
     * @return 注解
     */
    public static <T extends Annotation> List<T> getAnnotations(@NotNull Field field, @NotNull Class<T> annotationType) {
        Assert.notNull(field, "属性为空");
        Assert.notNull(annotationType, "注解类型为空");

        var clazz = field.getDeclaringClass();
        var fieldWrapper = getTypeWrapper(clazz).fields.get(field);
        if (fieldWrapper == null) {
            return Collections.emptyList();
        }

        return fieldWrapper.toCustomField().getAnnotations(annotationType);
    }

    /**
     * 获取属性的注解
     * <p>
     * 如果有多个，则返回第一个
     *
     * @param data           数据
     * @param fieldName      字段名称
     * @param annotationType 注解类型
     * @return 注解
     */
    public static <T extends Annotation> List<T> getAnnotations(@NotNull Object data, @NotBlank String fieldName, @NotNull Class<T> annotationType) {
        Assert.notNull(data, "数据为空");
        Assert.hasText(fieldName, "属性为空");
        Assert.notNull(annotationType, "注解类型为空");

        var fieldWrapper = getTypeWrapper(data).fieldNames.get(fieldName);
        if (fieldWrapper == null) {
            return Collections.emptyList();
        }

        return fieldWrapper.toCustomField().getAnnotations(annotationType);
    }

    /**
     * 获取读方法
     * <p>
     * 获取对象的get（或is）方法
     *
     * @param data  数据对象
     * @param field 属性
     * @return 方法
     */
    public static Method getReadMethod(@NotNull Object data, @NotNull Field field) {
        Assert.notNull(data, "数据为空");
        Assert.notNull(field, "属性为空");

        var fieldWrapper = getFieldByField(data, field);
        return fieldWrapper == null ? null : fieldWrapper.readMethod;
    }

    /**
     * 获取读方法
     * <p>
     * 获取对象的get（或is）方法
     *
     * @param data      数据对象
     * @param fieldName 属性名称
     * @return 方法
     */
    public static Method getReadMethod(@NotNull Object data, @NotBlank String fieldName) {
        Assert.notNull(data, "数据为空");
        Assert.hasText(fieldName, "属性名称为空");

        var fieldWrapper = getFieldByFieldName(data, fieldName);
        return fieldWrapper == null ? null : fieldWrapper.readMethod;
    }

    /**
     * 获取写方法
     * <p>
     * 获取对象的set方法
     *
     * @param data  数据对象
     * @param field 属性
     * @return 方法
     */
    public static Method getWriteMethod(@NotNull Object data, @NotNull Field field) {
        Assert.notNull(data, "数据为空");
        Assert.notNull(field, "属性为空");

        var fieldWrapper = getFieldByField(data, field);
        return fieldWrapper == null ? null : fieldWrapper.writeMethod;
    }

    /**
     * 获取读方法
     * <p>
     * 获取对象的set方法
     *
     * @param data      数据对象
     * @param fieldName 属性名称
     * @return 方法
     */
    public static Method getWriteMethod(@NotNull Object data, @NotBlank String fieldName) {
        Assert.notNull(data, "数据为空");
        Assert.hasText(fieldName, "属性名称为空");

        var fieldWrapper = getFieldByFieldName(data, fieldName);
        return fieldWrapper == null ? null : fieldWrapper.writeMethod;
    }

    private static FieldWrapper getFieldByFieldName(@NotNull Object data, String fieldName) {
        return getTypeWrapper(data).fieldNames.get(fieldName);
    }

    private static FieldWrapper getFieldByField(@NotNull Object data, Field field) {
        return getTypeWrapper(data).fields.get(field);
    }

    private static List<CustomField> getFields(@NotNull Object data, Class<? extends Annotation> annotation, boolean includeInherit) {
        var typeWrapper = getTypeWrapper(data);
        if (typeWrapper.simpleType) {
            return Collections.emptyList();
        }

        List<CustomField> fields = new ArrayList<>(8);
        for (Map.Entry<Field, FieldWrapper> entry : typeWrapper.fields.entrySet()) {
            // 不包含继承的
            if (!includeInherit && entry.getValue().inherit) {
                continue;
            }

            if (annotation == null) {
                fields.add(entry.getValue().toCustomField());
                continue;
            }

            // 根据注解获取
            if (entry.getValue().annotations.containsKey(annotation)) {
                fields.add(entry.getValue().toCustomField());
            }
        }
        return fields;
    }

    private static TypeWrapper getTypeWrapper(Object obj) {
        Class<?> clazz = obj instanceof Class ? (Class<?>) obj : obj.getClass();
        var typeWrapper = WRAPPER_MAP.get(clazz);
        if (typeWrapper == null) {
            typeWrapper = new TypeWrapper(clazz);
            WRAPPER_MAP.put(clazz, typeWrapper);
        }

        return typeWrapper;
    }

    static class TypeWrapper {
        private final Class<?> clazz;
        private boolean simpleType;
        private Map<Field, FieldWrapper> fields;
        private Map<String, FieldWrapper> fieldNames;

        TypeWrapper(Class<?> clazz) {
            Assert.notNull(clazz, "类型为空");
            this.clazz = clazz;

            this.analyze();
        }

        private void analyze() {
            this.simpleType = ClassUtil.isSimpleValueType(clazz);
            if (this.simpleType) {
                // 基本类型
                this.fields = Collections.emptyMap();
                this.fieldNames = Collections.emptyMap();
                return;
            }

            // 分析列
            var originalFields = ReflectUtil.getFields(clazz);
            if (ArrayUtil.isEmpty(originalFields)) {
                this.fields = Collections.emptyMap();
                this.fieldNames = Collections.emptyMap();
                return;
            }
            var propertyDesciptorMap = Arrays.stream(BeanUtils.getPropertyDescriptors(clazz))
                    .collect(Collectors.toMap(PropertyDescriptor::getName, Function.identity(), (t1, t2) -> t1));
            this.fields = new HashMap<>(originalFields.length);
            this.fieldNames = new HashMap<>(originalFields.length);
            for (Field field : originalFields) {
                if (Modifier.isStatic(field.getModifiers())) {
                    // 忽略静态属性
                    continue;
                }
                if (this.fieldNames.containsKey(field.getName())) {
                    // 出现同名属性时，优先使用子类的属性
                    continue;
                }

                var fieldWrapper = new FieldWrapper(clazz, field, propertyDesciptorMap);
                this.fields.put(field, fieldWrapper);
                this.fieldNames.put(field.getName(), fieldWrapper);
            }
        }
    }

    static class FieldWrapper {
        private final Field field;
        private boolean inherit;
        private Method readMethod;
        private Method writeMethod;
        private Map<Class<? extends Annotation>, List<Annotation>> annotations;

        FieldWrapper(Class<?> clazz, Field field, Map<String, PropertyDescriptor> propertyDescriptorMap) {
            Assert.notNull(field, "属性为空");
            this.field = field;

            this.analyze(clazz, propertyDescriptorMap);
        }

        private void analyze(Class<?> clazz, Map<String, PropertyDescriptor> propertyDescriptorMap) {
            // 是否是继承的属性
            this.inherit = field.getDeclaringClass() != clazz;

            // get set方法
            var propertyDescriptor = propertyDescriptorMap.get(field.getName());
            if (propertyDescriptor != null) {
                this.readMethod = propertyDescriptor.getReadMethod();
                this.writeMethod = propertyDescriptor.getWriteMethod();
            }

            // 注解
            var annotationsOriginal = field.getAnnotations();
            if (ArrayUtil.isEmpty(annotationsOriginal)) {
                this.annotations = Collections.emptyMap();
            } else {
                this.annotations = new HashMap<>(annotationsOriginal.length);
                for (var annotation : annotationsOriginal) {
                    this.annotations.computeIfAbsent(annotation.annotationType(), t -> new ArrayList<>(4))
                            .add(annotation);
                }
            }
        }

        public CustomField toCustomField() {
            return new CustomField(this);
        }
    }

    public static class CustomField {
        private final Field field;
        private final Method readMethod;
        private final Method writeMethod;
        private final Map<Class<? extends Annotation>, List<Annotation>> annotations;

        private CustomField() {
            this.field = null;
            this.readMethod = null;
            this.writeMethod = null;
            this.annotations = null;
        }

        private CustomField(FieldWrapper fieldWrapper) {
            this.field = fieldWrapper.field;
            this.readMethod = fieldWrapper.readMethod;
            this.writeMethod = fieldWrapper.writeMethod;
            this.annotations = this.cloneAnnotations(fieldWrapper.annotations);
        }

        private Map<Class<? extends Annotation>, List<Annotation>> cloneAnnotations(Map<Class<? extends Annotation>, List<Annotation>> source) {
            if (source.isEmpty()) {
                return Collections.emptyMap();
            }

            var target = new HashMap<Class<? extends Annotation>, List<Annotation>>();
            for (Map.Entry<Class<? extends Annotation>, List<Annotation>> entry : source.entrySet()) {
                target.put(entry.getKey(), new ArrayList<>(entry.getValue()));
            }
            return target;
        }

        public Field getField() {
            return field;
        }

        public Method getReadMethod() {
            return readMethod;
        }

        public Method getWriteMethod() {
            return writeMethod;
        }

        public Map<Class<? extends Annotation>, List<Annotation>> getAnnotations() {
            return annotations;
        }

        public String getName() {
            return field.getName();
        }

        public <T extends Annotation> T getAnnotation(@NotNull Class<T> annotationType) {
            var annotations = this.getAnnotations(annotationType);
            return annotations.isEmpty() ? null : annotations.get(0);
        }

        @SuppressWarnings("unchecked")
        public <T extends Annotation> List<T> getAnnotations(@NotNull Class<T> annotationType) {
            return (List<T>) annotations.getOrDefault(annotationType, Collections.emptyList());
        }

        @Override
        public String toString() {
            return field.getDeclaringClass().getName() + "." + field.getName();
        }
    }
}
