package com.elitescloud.boot.util;

import cn.hutool.core.lang.Assert;
import org.springframework.util.ClassUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Predicate;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/6/18
 */
public class ClassUtil extends cn.hutool.core.util.ClassUtil {

    /**
     * 获取bean的原始类
     *
     * @param bean bean对象
     * @return class
     */
    public static Class<?> getTargetClass(@NotNull Object bean) {
        return ProxyUtil.getTargetClass(bean);
    }

    /**
     * 获取泛型接口的泛型参数
     *
     * @param obj              对象或class
     * @param genericInterface 泛型接口
     * @return 参数列表
     */
    public static Type[] getTypeArguments(@NotNull Object obj, @NotNull Class<?> genericInterface) {
        Class<?> clazz = obj instanceof Class ? (Class<?>) obj : getTargetClass(obj);
        Assert.notNull(genericInterface, "泛型接口为空");

        while (true) {
            if (clazz == null || !genericInterface.isAssignableFrom(clazz)) {
                break;
            }
            var genericInterfaces = clazz.getGenericInterfaces();
            // 父类
            clazz = clazz.getSuperclass();
            if (ArrayUtil.isEmpty(genericInterfaces)) {
                continue;
            }

            Class<?> tempClazz = null;
            // 获取实现的接口
            for (Type inter : genericInterfaces) {
                var paramType = (ParameterizedType) inter;
                if (paramType.getRawType() == genericInterface) {
                    return paramType.getActualTypeArguments();
                }
                if (genericInterface.isAssignableFrom((Class<?>) paramType.getRawType()) && tempClazz == null) {
                    tempClazz = (Class<?>) paramType.getRawType();
                }
            }
            // 优先取父类
            if (clazz != null && !genericInterface.isAssignableFrom(clazz)) {
                clazz = tempClazz;
            }
        }
        return null;
    }

    /**
     * 是否存在类
     *
     * @param className 类名
     * @return 是否存在
     */
    public static boolean isPresent(@NotBlank String className) {
        return isPresent(className, null);
    }

    /**
     * 是否存在类
     *
     * @param className 类名
     * @return 是否存在
     */
    public static boolean isPresent(@NotBlank String className, ClassLoader classLoader) {
//        return ClassLoaderUtil.isPresent(className);

        if (classLoader == null) {
            classLoader = ClassUtils.getDefaultClassLoader();
        }
        try {
            Class.forName(className, false, classLoader);
            return true;
        } catch (Throwable ex) {
            return false;
        }
    }

    /**
     * 获取对象的类上包含指定注解的类
     *
     * @param bean            对象
     * @param annotationClass 注解类型
     * @return 类
     */
    public static Class<?> getClassWithAnnotation(@NotNull Object bean, @NotNull Class<? extends Annotation> annotationClass) {
        Assert.notNull(bean, "对象为空");
        return getClassWithAnnotation(getTargetClass(bean), annotationClass);
    }

    /**
     * 获取类上包含指定注解的类
     *
     * @param clazz           类
     * @param annotationClass 注解类型
     * @return 类
     */
    public static Class<?> getClassWithAnnotation(@NotNull Class<?> clazz, @NotNull Class<? extends Annotation> annotationClass) {
        Assert.notNull(annotationClass, "注解为空");

        List<Class<?>> classes = new ArrayList<>(8);
        findClassWithParents(clazz, classes, t -> t.isAnnotationPresent(annotationClass));

        return classes.isEmpty() ? null : classes.get(0);
    }

    /**
     * 获取对象的类上包含指定注解的类
     *
     * @param bean            对象
     * @param annotationClass 注解类型
     * @return 类
     */
    public static List<Class<?>> getClassesWithAnnotation(@NotNull Object bean, @NotNull Class<? extends Annotation> annotationClass) {
        Assert.notNull(bean, "对象为空");

        return getClassesWithAnnotation(getTargetClass(bean), annotationClass);
    }

    /**
     * 获取类上包含指定注解的类
     *
     * @param clazz           类
     * @param annotationClass 注解类型
     * @return 类
     */
    public static List<Class<?>> getClassesWithAnnotation(@NotNull Class<?> clazz, @NotNull Class<? extends Annotation> annotationClass) {
        Assert.notNull(clazz, "类为空");
        Assert.notNull(annotationClass, "注解为空");

        List<Class<?>> classes = new ArrayList<>(8);
        findClassWithParents(clazz, classes, t -> t.isAnnotationPresent(annotationClass));

        return classes;
    }

    /**
     * 获取方法的入参
     *
     * @param method 方法
     * @return 入参
     */
    public static Class<?>[] getMethodArgumentTypes(Method method) {
        Assert.notNull(method, "方法为空");

        return method.getParameterTypes();
    }

    /**
     * 获取方法的入参类型
     *
     * @param method 方法
     * @return 入参
     */
    public static String[] getMethodArgumentTypeNames(Method method) {
        Assert.notNull(method, "方法为空");

        var parameters = method.getParameterTypes();
        if (ArrayUtil.isEmpty(parameters)) {
            return new String[0];
        }

        return Arrays.stream(parameters).map(Class::getCanonicalName).toArray(String[]::new);
    }

    private static void findClassWithParents(@NotNull Class<?> clazz, @NotNull List<Class<?>> parents, Predicate<Class<?>> matcher) {
        if (clazz == null || clazz == Object.class) {
            return;
        }

        // 判断类是否满足
        if (matcher.test(clazz)) {
            parents.add(clazz);
            return;
        }

        // 判断接口是否满足
        var interfaces = clazz.getInterfaces();
        if (ArrayUtil.isNotEmpty(interfaces)) {
            for (Class<?> anInterface : interfaces) {
                findClassWithParents(anInterface, parents, matcher);
                if (!parents.isEmpty()) {
                    parents.add(clazz);
                    return;
                }
            }
        }

        // 判断父类是否满足
        findClassWithParents(clazz.getSuperclass(), parents, matcher);
        if (!parents.isEmpty()) {
            parents.add(clazz);
        }
    }
}
