package com.elitescloud.boot.util;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.util.encrypt.BaseEncrypt;
import org.bouncycastle.asn1.ASN1BitString;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.pkcs.RSAPrivateKey;
import org.bouncycastle.asn1.pkcs.RSAPublicKey;
import org.springframework.util.Assert;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPublicKeySpec;

/**
 * RSA工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/5
 */
public class RsaUtil extends BaseEncrypt {

    /**
     * 加密算法
     */
    private static final String KEY_ALGORITHM = "RSA";

    /**
     * 签名算法
     */
    private static final String DEFAULT_SIGNATURE_ALGORITHM = "SHA256WithRSA";

    /**
     * 加密算法、工作模式和填充模式
     */
    private static final String DEFAULT_CIPHER_TYPE = "RSA/ECB/OAEPPadding";

    /**
     * 加密时分段大小
     */
    private static final int DEFAULT_LEN_ENCRYPT = 245;

    /**
     * 解密时分段大小
     */
    private static final int DEFAULT_LEN_DECRYPT = 256;

    /**
     * 生成密钥对
     *
     * @return 密钥对
     */
    public static KeyPair generateKeyPair() {
        return generateKeyPair(2048);
    }

    /**
     * 生成密钥对
     *
     * @param keySize 秘钥大小，取值如1024、2048、4096等，取值越大，安全性越高，同时计算开销也会增大
     * @return 米哟对
     */
    public static KeyPair generateKeyPair(int keySize) {
        var random = generateSecureRandom(DEFAULT_RANDOM_TYPE);
        return generateKeyPair(KEY_ALGORITHM, keySize, random);
    }

    /**
     * 秘钥转字符串
     *
     * @param key 秘钥
     * @return 字符串
     */
    public static String convertKey2Str(@NotNull Key key) {
        Assert.notNull(key, "秘钥为空");
        return encodeBase64Str(key.getEncoded());
    }

    /**
     * 公钥字符串转公钥
     *
     * @param key 公钥字符串
     * @return 公钥
     */
    public static PublicKey convert2PublicKey(@NotBlank String key) {
        KeyFactory keyFactory = generateKeyFactory(KEY_ALGORITHM);
        return convert2PublicKey(key, keyFactory);
    }

    /**
     * 私钥字符串转私钥
     *
     * @param key 私钥字符串
     * @return 私钥
     */
    public static PrivateKey convert2PrivateKey(@NotBlank String key) {
        KeyFactory keyFactory = generateKeyFactory(KEY_ALGORITHM);
        return convert2PrivateKey(key, keyFactory);
    }

    /**
     * 转换公钥格式
     * <p>
     * pkcs#8转pkcs#1
     *
     * @param publicKey 公钥
     * @return pkcs#1格式
     */
    public static byte[] convert2Pkcs1(@NotNull PublicKey publicKey) {
        return ASN1BitString.getInstance(ASN1Sequence.getInstance(publicKey.getEncoded()).getObjectAt(1)).getBytes();
    }

    /**
     * 转换公钥格式
     * <p>
     * pkcs#1转pkcs8
     *
     * @param key 公钥内容
     * @return 公钥
     * @throws Exception 转换异常
     */
    public static PublicKey convert2PublicKeyForPkcs1(@NotBlank String key) throws Exception {
        Assert.hasText(key, "公钥为空");

        KeyFactory keyFactory = generateKeyFactory(KEY_ALGORITHM);
        byte[] keys = decodeBase64(key);

        try {
            RSAPublicKey rsaPublicKey = RSAPublicKey.getInstance(keys);
            RSAPublicKeySpec rsaPublicKeySpec = new RSAPublicKeySpec(rsaPublicKey.getModulus(), rsaPublicKey.getPublicExponent());
            return keyFactory.generatePublic(rsaPublicKeySpec);
        } catch (Exception e) {
            throw new IllegalStateException("公钥无效：", e);
        }
    }

    /**
     * 转换私钥格式
     * <p>
     * pkcs#8转pkcs#1
     *
     * @param privateKey 私钥
     * @return pkcs#1格式
     */
    public static byte[] convert2Pkcs1(@NotNull PrivateKey privateKey) throws IOException {
        return PrivateKeyInfo.getInstance(privateKey.getEncoded()).parsePrivateKey().toASN1Primitive().getEncoded();
    }

    /**
     * 转换私钥格式
     * <p>
     * pkcs#1转pkcs8
     *
     * @param key 私钥内容
     * @return 私钥
     * @throws Exception 转换异常
     */
    public static PrivateKey convert2PrivateKeyForPkcs1(@NotBlank String key) throws Exception {
        Assert.hasText(key, "私钥为空");

        KeyFactory keyFactory = generateKeyFactory(KEY_ALGORITHM);
        byte[] keys = decodeBase64(key);

        try {
            RSAPrivateKey rsaPrivateKey = RSAPrivateKey.getInstance(keys);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(rsaPrivateKey.getEncoded());

            return keyFactory.generatePrivate(keySpec);
        } catch (Exception e) {
            throw new IllegalStateException("私钥无效：", e);
        }
    }

    /**
     * 签名
     *
     * @param privateKey 私钥
     * @param plainText  待签名内容
     * @return 签名
     */
    public static String sign(@NotNull PrivateKey privateKey, String signatureAlgorithm, @NotBlank String plainText) {
        Signature signature = generateSignature(CharSequenceUtil.blankToDefault(signatureAlgorithm, DEFAULT_SIGNATURE_ALGORITHM));
        return sign(privateKey, signature, plainText);
    }

    /**
     * 验证签名
     *
     * @param publicKey 公钥
     * @param plainText 明文
     * @param sign      签名
     * @return 是否验证通过
     */
    public static boolean verifySign(@NotNull PublicKey publicKey, String signatureAlgorithm, @NotBlank String plainText, @NotBlank String sign) {
        Signature signature = generateSignature(CharSequenceUtil.blankToDefault(signatureAlgorithm, DEFAULT_SIGNATURE_ALGORITHM));
        return verifySign(publicKey, signature, plainText, sign);
    }

    /**
     * 加密
     *
     * @param publicKey 公钥
     * @param plainText 明文
     * @return 密文
     */
    public static String encrypt(@NotNull PublicKey publicKey, String cipherType, @NotBlank String plainText) {
        try {
            return encrypt(publicKey, null, null, CharSequenceUtil.blankToDefault(cipherType, DEFAULT_CIPHER_TYPE), plainText);
        } catch (Exception e) {
            throw new RuntimeException("数据加密失败：", e);
        }
    }

    /**
     * 解密
     *
     * @param privateKey 私钥
     * @param ciphertext 密文
     * @return 明文
     */
    public static String decrypt(@NotNull PrivateKey privateKey, String cipherType, @NotBlank String ciphertext) {
        try {
            return decrypt(privateKey, null, null, CharSequenceUtil.blankToDefault(cipherType, DEFAULT_CIPHER_TYPE), ciphertext);
        } catch (Exception e) {
            throw new RuntimeException("数据解密失败：", e);
        }
    }
}
