package com.elitescloud.boot.util.encrypt;

import cn.hutool.core.text.CharSequenceUtil;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.*;
import org.bouncycastle.operator.OutputEncryptor;
import org.bouncycastle.pkcs.PKCS8EncryptedPrivateKeyInfo;
import org.bouncycastle.util.io.pem.PemWriter;
import org.springframework.core.io.Resource;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;

import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * 加密工具基础类.
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/5
 */
public abstract class BaseEncrypt {

    /**
     * 编码类型
     */
    protected static final Charset CHARSET = StandardCharsets.UTF_8;

    /**
     * 随机类型
     * <p>
     * 如SHA1PRNG、NativePRNG等
     */
    protected static final String DEFAULT_RANDOM_TYPE = "SHA1PRNG";

    private static final Provider PROVIDER = new BouncyCastleProvider();

    static {
        Security.addProvider(PROVIDER);
    }

    /**
     * byte数组转字符串
     *
     * @param data byte数组
     * @return 字符串
     */
    public static String encodeBase64Str(byte[] data) {
        var bytes = org.bouncycastle.util.encoders.Base64.encode(data);
        return new String(bytes, CHARSET);
    }

    /**
     * 字符串转byte数组
     *
     * @param data 字符串
     * @return byte数组
     */
    public static byte[] decodeBase64(String data) {
        try {
            return org.bouncycastle.util.encoders.Base64.decode(data.getBytes(CHARSET));
        } catch (Exception e) {
            throw new IllegalArgumentException("Base64解码失败", e);
        }
    }

    /**
     * 生成安全随机因子
     *
     * @param secureRandomAlgorithm 随机因子算法
     * @return 安全随机因子
     */
    protected static SecureRandom generateSecureRandom(@NotBlank String secureRandomAlgorithm) {
        Assert.hasText(secureRandomAlgorithm, "随机因子算法名称为空");

        try {
            return SecureRandom.getInstance(secureRandomAlgorithm);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("生成安全随机因子异常，请确认支持算法" + secureRandomAlgorithm, e);
        }
    }

    /**
     * 生成密钥对
     *
     * @param keyAlgorithm 算法类型
     * @param keySize      秘钥大小
     * @param secureRandom 随机因子
     * @return 密钥对
     */
    protected static KeyPair generateKeyPair(@NotBlank String keyAlgorithm, int keySize, @NotNull SecureRandom secureRandom) {
        Assert.hasText(keyAlgorithm, "秘钥算法名称为空");
        Assert.notNull(secureRandom, "随机因子为空");

        KeyPairGenerator generator = null;
        try {
            generator = KeyPairGenerator.getInstance(keyAlgorithm);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("生成密钥对失败，请确认支持加密算法" + keyAlgorithm);
        }

        generator.initialize(keySize, secureRandom);
        return generator.generateKeyPair();
    }

    /**
     * 获取秘钥工厂
     *
     * @param keyAlgorithm 算法类型
     * @return 秘钥工厂
     */
    protected static KeyFactory generateKeyFactory(@NotBlank String keyAlgorithm) {
        Assert.hasText(keyAlgorithm, "秘钥算法名称为空");

        try {
            return KeyFactory.getInstance(keyAlgorithm);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("读取公钥失败，请确认是否支持" + keyAlgorithm, e);
        }
    }

    /**
     * 公钥字符串转公钥
     *
     * @param key 公钥字符串
     * @return 公钥
     */
    protected static PublicKey convert2PublicKey(@NotBlank String key, @NotNull KeyFactory keyFactory) {
        Assert.hasText(key, "公钥为空");
        Assert.notNull(keyFactory, "公钥工厂为空");

        key = key.replace("-----BEGIN PUBLIC KEY-----", "").replace("-----END PUBLIC KEY-----", "").trim();
        byte[] keys = decodeBase64(key);
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(keys);

        try {
            return keyFactory.generatePublic(keySpec);
        } catch (InvalidKeySpecException e) {
            throw new IllegalStateException("公钥无效：", e);
        }
    }

    /**
     * 私钥字符串转私钥
     *
     * @param key 私钥字符串
     * @return 私钥
     */
    protected static PrivateKey convert2PrivateKey(@NotBlank String key, @NotNull KeyFactory keyFactory) {
        Assert.hasText(key, "私钥为空");

        key = key.replace("-----BEGIN ENCRYPTED PRIVATE KEY-----", "").replace("-----END ENCRYPTED PRIVATE KEY-----", "").trim();
        byte[] keys = decodeBase64(key);
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keys);

        try {
            return keyFactory.generatePrivate(keySpec);
        } catch (InvalidKeySpecException e) {
            throw new IllegalStateException("私钥无效：", e);
        }
    }

    /**
     * 转私钥至pem文件
     *
     * @param privateKey 私钥
     * @param password   获取私钥时的密码
     * @return 含有私钥的pem文件
     * @throws Exception 异常
     */
    public static ByteArrayOutputStream convertPrivateKey2Pem(@NotNull PrivateKey privateKey, String password) throws Exception {
        Assert.notNull(privateKey, "私钥不能为空");

        OutputEncryptor encryptor = null;
        if (CharSequenceUtil.isNotBlank(password)) {
            encryptor = new JceOpenSSLPKCS8EncryptorBuilder(PKCSObjectIdentifiers.pbeWithSHAAnd3_KeyTripleDES_CBC)
                    .setProvider(PROVIDER)
                    .setRandom(SecureRandom.getInstance(DEFAULT_RANDOM_TYPE))
                    .setPassword(password.toCharArray())
                    .build();
        }

        var outputStream = new ByteArrayOutputStream();
        try (PemWriter writer = new PemWriter(new OutputStreamWriter(outputStream, CHARSET))) {
            writer.writeObject(new JcaPKCS8Generator(privateKey, encryptor));
        }
        return outputStream;
    }

    /**
     * 转换公钥文件
     *
     * @param publicKey 公钥
     * @return 字节流
     */
    public static ByteArrayOutputStream convertPublicKey(@NotNull PublicKey publicKey) throws Exception {
        Assert.notNull(publicKey, "公钥不能为空");

        var outputStream = new ByteArrayOutputStream();
        try (JcaPEMWriter writer = new JcaPEMWriter(new OutputStreamWriter(outputStream, CHARSET))) {
            writer.writeObject(publicKey);
        }

        return outputStream;
    }

    /**
     * 转换私钥字符串
     *
     * @param inputStream 输入流
     * @param password    私钥密码
     * @return 私钥
     * @throws Exception 异常
     */
    public static PrivateKey convertPrivateKeyFromPem(@NotNull InputStream inputStream, @NotBlank String password) throws Exception {
        Assert.notNull(inputStream, "私钥不能为空");
        Assert.notNull(password, "私钥密码不能为空");

        var descriptor = new JceOpenSSLPKCS8DecryptorProviderBuilder().setProvider(PROVIDER).build(password.toCharArray());

        PEMParser reader = new PEMParser(new InputStreamReader(inputStream));
        PKCS8EncryptedPrivateKeyInfo privateKeyInfo = (PKCS8EncryptedPrivateKeyInfo) reader.readObject();

        return new JcaPEMKeyConverter().setProvider(PROVIDER).getPrivateKey(privateKeyInfo.decryptPrivateKeyInfo(descriptor));
    }

    /**
     * 公钥字符串转公钥
     *
     * @param inputStream 公钥输入流
     * @return 公钥
     * @throws Exception 异常信息
     */
    public static PublicKey convertPublicKeyFromPem(@NotNull InputStream inputStream) throws Exception {
        Assert.notNull(inputStream, "数据为空");

        PEMParser parser = new PEMParser(new InputStreamReader(inputStream));

        SubjectPublicKeyInfo publicKeyInfo = (SubjectPublicKeyInfo) parser.readObject();
        var convert = new JcaPEMKeyConverter().setProvider(PROVIDER);
        return convert.getPublicKey(publicKeyInfo);
    }

    /**
     * 从秘钥库加载公钥
     *
     * @param keyStore 秘钥库
     * @param alias    别名
     * @return 公钥
     */
    public static PublicKey loadPublicKey(@NotNull KeyStore keyStore, @NotBlank String alias) throws Exception {
        Assert.notNull(keyStore, "秘钥库为空");
        Assert.hasText(alias, "别名为空");

        return keyStore.getCertificate(alias).getPublicKey();
    }

    /**
     * 从秘钥库加载私钥
     *
     * @param keyStore 秘钥库
     * @param alias    别名
     * @return 私钥
     */
    public static PrivateKey loadPrivateKey(@NotNull KeyStore keyStore, @NotBlank String alias, @NotBlank String password) throws Exception {
        Assert.notNull(keyStore, "秘钥库为空");
        Assert.hasText(alias, "别名为空");
        Assert.hasText(password, "密码为空");

        return (PrivateKey) keyStore.getKey(alias, password.toCharArray());
    }

    /**
     * 加载KeyStore工具
     *
     * @param keystoreStream keyStore流
     * @param type           keyStore类型
     * @param password       keystore密码
     * @param alias          key别名
     * @param secret         key secret
     * @return KeyStore
     */
    public static KeyStore loadKeystore(@NonNull InputStream keystoreStream, @NotBlank String type, @NotBlank String password,
                                        @NotBlank String alias, @NotBlank String secret) throws Exception {
        Assert.hasText(password, "password为空");
        Assert.hasText(type, "秘钥库类型为空");
        Assert.hasText(password, "密码为空");
        Assert.hasText(alias, "alias为空");
        Assert.hasText(secret, "secret为空");

        KeyStore keyStore = KeyStore.getInstance(type);
        keyStore.load(keystoreStream, password.toCharArray());

        return keyStore;
    }

    /**
     * 加载KeyStore工具
     *
     * @param keystoreResource keyStore资源
     * @param type             keyStore类型
     * @param password         keystore密码
     * @param alias            key别名
     * @param secret           key secret
     * @return KeyStore
     */
    public static KeyStore loadKeystore(@NonNull Resource keystoreResource, String type, @NonNull String password, String alias, String secret) throws Exception {
        Assert.isTrue(keystoreResource.exists(), "keystore不存在");

        InputStream keystoreStream = keystoreResource.getInputStream();
        return loadKeystore(keystoreStream, type, password, alias, secret);
    }

    /**
     * 生成签名算法
     *
     * @param signatureAlgorithm 签名算法名称
     * @return 签名算法
     */
    protected static Signature generateSignature(@NotBlank String signatureAlgorithm) {
        Assert.hasText(signatureAlgorithm, "签名算法名称为空");

        try {
            return Signature.getInstance(signatureAlgorithm);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("获取签名算法失败，请确认是否支持" + signatureAlgorithm, e);
        }
    }

    /**
     * 签名
     *
     * @param privateKey 私钥
     * @param signature  签名算法
     * @param plainText  待签名内容
     * @return 签名
     */
    public static String sign(@NotNull PrivateKey privateKey, @NotNull Signature signature, @NotBlank String plainText) {
        Assert.notNull(privateKey, "私钥为空");
        Assert.notNull(signature, "签名算法为空");
        Assert.notNull(plainText, "待签名内容为空");

        try {
            signature.initSign(privateKey);
        } catch (InvalidKeyException e) {
            throw new IllegalStateException("签名数据失败，私钥无效：", e);
        }

        try {
            signature.update(plainText.getBytes(CHARSET));
        } catch (SignatureException e) {
            throw new IllegalStateException("签名数据失败：", e);
        }

        try {
            return encodeBase64Str(signature.sign());
        } catch (SignatureException e) {
            throw new IllegalStateException("签名数据失败：", e);
        }
    }

    /**
     * 验证签名
     *
     * @param publicKey 公钥
     * @param signature 签名算法
     * @param plainText 明文
     * @param sign      签名
     * @return 是否验证通过
     */
    public static boolean verifySign(@NotNull PublicKey publicKey, @NotNull Signature signature, @NotBlank String plainText, @NotBlank String sign) {
        Assert.notNull(publicKey, "公钥为空");
        Assert.notNull(signature, "签名算法为空");
        Assert.notNull(plainText, "明文内容为空");
        Assert.notNull(sign, "签名为空");
        try {
            signature.initVerify(publicKey);
        } catch (InvalidKeyException e) {
            throw new IllegalStateException("签名数据失败，公钥无效：", e);
        }
        try {
            signature.update(plainText.getBytes(CHARSET));
        } catch (SignatureException e) {
            throw new IllegalStateException("验证签名失败：", e);
        }

        try {
            return signature.verify(decodeBase64(sign));
        } catch (SignatureException e) {
            throw new IllegalStateException("验证签名失败：", e);
        }
    }

    /**
     * 加密
     *
     * @param key        秘钥
     * @param blockSize  分段大小
     * @param iv         偏移量
     * @param cipherType 模式
     * @param plainText  明文
     * @return 密文
     */
    protected static String encrypt(Key key, Integer blockSize, String iv, String cipherType, String plainText) throws Exception {
        Cipher cipher = Cipher.getInstance(cipherType);

        if (iv == null) {
            cipher.init(Cipher.ENCRYPT_MODE, key);
        } else {
            cipher.init(Cipher.ENCRYPT_MODE, key, new IvParameterSpec(iv.getBytes(CHARSET)));
        }

        byte[] data = plainText.getBytes(CHARSET);
        if (blockSize == null) {
            return encodeBase64Str(cipher.doFinal(data));
        }

        // 分段加密
        byte[] result = cipherBySplit(cipher, data, blockSize);
        return encodeBase64Str(result);
    }

    /**
     * 解密
     *
     * @param key        秘钥
     * @param blockSize  分段大小
     * @param iv         偏移量
     * @param cipherType 模式
     * @param cipherText 密文
     * @return 明文
     */
    protected static String decrypt(Key key, Integer blockSize, String iv, String cipherType, String cipherText) throws Exception {
        Cipher cipher = Cipher.getInstance(cipherType);

        if (iv == null) {
            cipher.init(Cipher.DECRYPT_MODE, key);
        } else {
            cipher.init(Cipher.DECRYPT_MODE, key, new IvParameterSpec(iv.getBytes(CHARSET)));
        }

        byte[] data = decodeBase64(cipherText);
        if (blockSize == null) {
            return new String(cipher.doFinal(decodeBase64(cipherText)), CHARSET);
        }

        // 分段解密
        byte[] result = cipherBySplit(cipher, data, blockSize);
        return new String(result, CHARSET);
    }

    private static byte[] cipherBySplit(Cipher cipher, byte[] data, Integer blockSize) throws Exception {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int offset = 0;
        int dataLen = data.length;
        int size = 0;
        byte[] cache;
        while (true) {
            size = Math.min(blockSize, dataLen - offset);
            cache = cipher.doFinal(data, offset, size);
            out.write(cache);

            offset += blockSize;
            if (offset >= dataLen) {
                break;
            }
        }
        return out.toByteArray();
    }
}
