package com.elitescloud.boot.exception;

import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ErrorDetail;
import com.elitescloud.cloudt.common.base.ErrorDisplayType;
import com.elitescloud.cloudt.common.base.ErrorSource;
import org.apache.skywalking.apm.toolkit.trace.IgnoredException;

import java.util.function.Supplier;

/**
 * 业务异常 (BusinessException)
 * 这类异常涉及业务逻辑，例如用户尝试购买一个已经售完的产品。
 * 这类异常日志级别是error
 * <p>
 * 校验异常 (ValidationException)
 * 这类异常涉及输入的校验，例如用户输入了一个无效的电子邮箱地址。
 * 这类日志级别是WARN用于校验异常。例如，用户输入格式错误，但这并不会导致系统崩溃
 * <p>
 * 系统异常 (SystemException)
 * 这类异常涉及底层系统或运行时的问题，例如数据库连接失败。
 * 这类日志级别error用于系统异常。这表示严重的问题，可能导致系统的部分或全部崩溃。
 * <p>
 * 错误来源
 * A 表示错误来源于用户 ，比如参数错误 ，用户安装版本过低 ，用户支付超时等问题；
 * B 表示错误来源于当前系统 ，往往是业务逻辑出错 ，或程序健壮性差等问题；
 * C 表示错误来源于第三方服务 ，比如 CDN
 *
 * @author Michael Li
 * @date 2020/6/22
 */
@IgnoredException
public class BusinessException extends RuntimeException implements Supplier<BusinessException> {

    private static final long serialVersionUID = -2205002357611194846L;

    /**
     * 系统错误码
     */
    private ApiCode apiCode;
    /**
     * 自定义错误码
     * <p>
     * 与 {@link #apiCode} 二选一即可
     */
    private Integer code;
    /**
     * 错误来源
     */
    private ErrorSource source;
    /**
     * 显示类型
     */
    private ErrorDisplayType displayType;
    /**
     * 详细错误信息
     */
    private ErrorDetail detail;

    public static BusinessException instance() {
        return new BusinessException();
    }

    public BusinessException() {
        super();
        this.apiCode = ApiCode.BUSINESS_EXCEPTION;
        this.code = null;
        this.source = ErrorSource.B;
        this.displayType = ErrorDisplayType.REMINDER;
        this.detail = null;
    }

    public BusinessException(ErrorSource source, String message, ErrorDisplayType displayType,
                             ErrorDetail detail) {
        super(message);
        this.apiCode = ApiCode.BUSINESS_EXCEPTION;
        this.code = null;
        this.source = source;
        this.displayType = displayType;
        this.detail = detail;
    }

    public BusinessException(String message, Throwable cause, ApiCode apiCode, Integer code, ErrorSource source, ErrorDisplayType displayType, ErrorDetail detail) {
        super(message, cause);
        this.apiCode = apiCode;
        this.code = code;
        this.source = source;
        this.displayType = displayType;
        this.detail = detail;
    }

    public BusinessException(String message) {
        super(message);
        this.apiCode = ApiCode.BUSINESS_EXCEPTION;
        this.code = null;
        this.source = ErrorSource.D;
        this.displayType = ErrorDisplayType.REMINDER;
        this.detail = null;
    }

    public BusinessException(ApiCode apiCode, String message) {
        super(message);
        this.apiCode = apiCode;
        this.code = null;
        this.source = ErrorSource.D;
        this.displayType = ErrorDisplayType.REMINDER;
        this.detail = null;
    }

    public BusinessException(ApiCode apiCode, String message, Throwable throwable) {
        super(message, throwable);
        this.apiCode = apiCode;
        this.code = null;
        this.source = ErrorSource.D;
        this.displayType = ErrorDisplayType.REMINDER;
        this.detail = null;
    }

    public BusinessException(Integer code, String message) {
        super(message);
        this.apiCode = null;
        this.code = code;
        this.source = ErrorSource.D;
        this.displayType = ErrorDisplayType.REMINDER;
        this.detail = null;
    }

    public BusinessException(Integer code, String message, Throwable throwable) {
        super(message, throwable);
        this.apiCode = null;
        this.code = code;
        this.source = ErrorSource.D;
        this.displayType = ErrorDisplayType.REMINDER;
        this.detail = null;
    }

    public BusinessException(String message, Throwable cause) {
        super(message, cause);
        this.apiCode = null;
        this.code = null;
        this.source = ErrorSource.D;
        this.displayType = ErrorDisplayType.REMINDER;
        this.detail = null;
    }

    public BusinessException(Throwable cause) {
        super(cause);
        this.apiCode = null;
        this.code = null;
        this.source = ErrorSource.D;
        this.displayType = ErrorDisplayType.REMINDER;
        this.detail = null;
    }

    public BusinessException(String message, Throwable cause, boolean enableSuppression, boolean writableStackTrace) {
        super(message, cause, enableSuppression, writableStackTrace);
        this.apiCode = null;
        this.code = null;
        this.source = ErrorSource.D;
        this.displayType = ErrorDisplayType.REMINDER;
        this.detail = null;
    }

    @Override
    public BusinessException get() {
        return this;
    }

    public ApiCode getApiCode() {
        return apiCode;
    }

    public ErrorSource getSource() {
        return source;
    }

    public ErrorDisplayType getDisplayType() {
        return displayType;
    }

    public ErrorDetail getDetail() {
        return detail;
    }

    public Integer getCode() {
        if (code == null && apiCode != null) {
            return apiCode.getCode();
        }
        return code;
    }

    public BusinessException setApiCode(ApiCode apiCode) {
        this.apiCode = apiCode;
        return this;
    }

    public BusinessException setCode(Integer code) {
        this.code = code;
        return this;
    }

    public BusinessException setSource(ErrorSource source) {
        this.source = source;
        return this;
    }

    public BusinessException setDisplayType(ErrorDisplayType displayType) {
        this.displayType = displayType;
        return this;
    }

    public BusinessException setDetail(ErrorDetail detail) {
        this.detail = detail;
        return this;
    }
}
