package com.elitescloud.cloudt.tims.sso;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.authorization.sdk.util.ObjectMapperFactory;
import com.elitescloud.cloudt.authorization.sdk.util.RestTemplateFactory;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.tims.sso.common.TimsSsoConstant;
import com.elitescloud.cloudt.tims.sso.params.TimsSsoLoginDTO;
import com.elitescloud.cloudt.tims.sso.params.TimsSsoResetPwdDTO;
import com.elitescloud.cloudt.tims.sso.params.TimsSsoResult;
import com.elitescloud.cloudt.tims.sso.params.TimsSsoUserDTO;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.net.URI;
import java.util.Objects;

/**
 * SSO客户端.
 *
 * @author Kaiser（wang shao）
 * @date 2023/7/26
 */
@Slf4j
public class SsoSdkClient {

    private TimsSsoProperties ssoProperties;
    private final RestTemplate restTemplate;
    private final ObjectMapper objectMapper;

    /**
     * 客户端实例
     */
    private static SsoSdkClient instance;

    public static SsoSdkClient getInstance() {
        if (instance == null) {
            synchronized (SsoSdkClient.class) {
                if (instance != null) {
                    return instance;
                }
                instance = new SsoSdkClient();
            }
        }
        return instance;
    }

    private SsoSdkClient() {
        this.restTemplate = RestTemplateFactory.instance();
        this.objectMapper = ObjectMapperFactory.instance();
    }

    /**
     * 新增用户
     *
     * @param user 用户信息
     */
    public String addUser(@NotNull TimsSsoUserDTO user) {
        // 检查信息完整性
        this.checkUser(user);

        if (log.isDebugEnabled()) {
            log.debug("[SSO]同步用户信息：{}", this.obj2json(user));
        }

        // 调用接口新增数据
        TimsSsoResult result = this.exchange(TimsSsoConstant.URI_USER_ADD, HttpMethod.POST, new HttpEntity<>(user), new ParameterizedTypeReference<>() {
        });
        log.info("[SSO]同步用户信息结果：{}", result);

        return result.getDataSafely(errCode -> {
            String prefix = "向SSO同步账号信息失败";
            if (errCode == TimsSsoConstant.CODE_USER_EXISTS) {
                return prefix + ":" + (StringUtils.hasText(user.getPhonenumber()) ? "手机号已存在" : "账号已存在");
            }
            return prefix;
        });
    }

    /**
     * 通过登录获取token
     *
     * @param loginDTO 登录参数
     * @return token
     */
    public String tokenByLogin(@NotNull TimsSsoLoginDTO loginDTO) {
        Assert.hasText(loginDTO.getUsername(), "用户名为空");
        Assert.hasText(loginDTO.getPassword(), "密码为空");

        loginDTO.setBrandCode(ssoProperties.getBrandCode());

        if (log.isDebugEnabled()) {
            log.debug("[SSO]用户登录信息：{}", this.obj2json(loginDTO));
        }

        // 调用接口生成token
        TimsSsoResult result = this.exchange(TimsSsoConstant.URI_USER_TOKEN_LOGIN, HttpMethod.POST, new HttpEntity<>(loginDTO), new ParameterizedTypeReference<>() {
        });
        log.info("[SSO]用户登录结果：{}", result);
        if (Objects.equals(result.getCode(), TimsSsoConstant.CODE_NO_PERMISSION)) {
            log.warn("不支持的账号类型，账号：{}", loginDTO.getUsername());
            return null;
        }

        var token = result.getDataSafely(errCode -> {
            if (errCode == TimsSsoConstant.CODE_LOGIN_PWD_ERROR) {
                return "账号或密码错误";
            }
            return "SSO认证失败";
        });
        if (CharSequenceUtil.startWithIgnoreCase(token, TimsSsoConstant.TOKEN_TYPE)) {
            token = token.substring(TimsSsoConstant.TOKEN_TYPE.length());
        }
        return token.trim();
    }

    /**
     * token续期
     *
     * @param token token
     * @return 是否成功
     */
    public boolean tokenByRenew(@NotBlank String token) {
        Assert.hasText(token, "token为空");

        if (!CharSequenceUtil.startWithIgnoreCase(token, TimsSsoConstant.TOKEN_TYPE)) {
            token = TimsSsoConstant.TOKEN_TYPE + " " + token;
        }
        log.info("[SSO]token续期：{}", token);

        HttpHeaders headers = new HttpHeaders();
        headers.add(HttpHeaders.AUTHORIZATION, token);

        // 调用接口生成token
        TimsSsoResult result = this.exchange(TimsSsoConstant.URI_USER_TOKEN_RENEW, HttpMethod.GET, new HttpEntity<>(headers), new ParameterizedTypeReference<>() {
        });
        log.info("[SSO]token续期结果：{}", result);

        return result.isSuccess();
    }

    /**
     * 重置密码
     *
     * @param resetPwdDTO 重置密码信息
     */
    public boolean resetPassword(@NotNull TimsSsoResetPwdDTO resetPwdDTO) {
        // 检查信息完整性
        this.checkRestPwd(resetPwdDTO);

        if (log.isDebugEnabled()) {
            log.debug("[SSO]重置密码信息：{}", this.obj2json(resetPwdDTO));
        }

        // 调用接口新增数据
        TimsSsoResult result = this.exchange(TimsSsoConstant.URI_USER_PASSWORD_RESET, HttpMethod.PUT, new HttpEntity<>(resetPwdDTO), new ParameterizedTypeReference<>() {
        });
        log.info("[SSO]重置密码结果：{}", result);

        return result.isSuccess();
    }

    void setSsoProperties(TimsSsoProperties ssoProperties) {
        instance.ssoProperties = ssoProperties;
        Assert.hasText(ssoProperties.getServerAddr(), "SSO服务的地址未配置");
        Assert.hasText(ssoProperties.getBrandCode(), "SSO的品牌编号未配置");
    }

    private void checkUser(TimsSsoUserDTO user) {
        Assert.notNull(user, "用户信息为空");
        Assert.hasText(user.getUserType(), "用户类型为空");

        if (TimsSsoConstant.USER_TYPE_HEADQUARTERS.equals(user.getUserType())) {
            // 总部员工
            Assert.hasText(user.getEmployeeCode(), "员工号为空");
        } else if (TimsSsoConstant.USER_TYPE_PARTNER.equals(user.getUserType())) {
            // 加盟商员工
            Assert.hasText(user.getPhonenumber(), "手机号为空");
            Assert.hasText(user.getUsername(), "用户名为空");
        } else {
            throw new IllegalArgumentException("不支持的账号类型：" + user.getUserType());
        }

        Assert.hasText(user.getPassword(), "密码为空");

        user.setBrandCode(ssoProperties.getBrandCode());
    }

    private void checkRestPwd(TimsSsoResetPwdDTO resetPwdDTO) {
        Assert.notNull(resetPwdDTO, "重置密码信息为空");

        Assert.hasText(resetPwdDTO.getUserType(), "用户类型为空");
        if (!CharSequenceUtil.equalsAny(resetPwdDTO.getUserType(), TimsSsoConstant.USER_TYPE_HEADQUARTERS,
                TimsSsoConstant.USER_TYPE_PARTNER)) {
            throw new IllegalArgumentException("不支持的账号类型：" + resetPwdDTO.getUserType());
        }
        Assert.hasText(resetPwdDTO.getUsername(), "用户名为空");
        Assert.hasText(resetPwdDTO.getOldPassword(), "旧密码为空");
        Assert.hasText(resetPwdDTO.getNewPassword(), "新密码为空");

        resetPwdDTO.setBrandCode(ssoProperties.getBrandCode());
    }

    private <T> T exchange(@NotNull String uri, @NotNull HttpMethod httpMethod, HttpEntity<?> httpEntity, ParameterizedTypeReference<T> responseType, String... uriParameters) {
        Assert.notNull(ssoProperties, "SSO配置尚未初始化");
        String url = this.buildURI(uri, uriParameters).toString();
        ResponseEntity<T> response = null;

        try {
            response = this.restTemplate.exchange(url, httpMethod, httpEntity, responseType);
        } catch (Throwable var8) {
            log.error("[SSO]{}调用失败：", url, var8);
            throw new IllegalStateException("远程服务器异常", var8);
        }

        if (response.getStatusCode() != HttpStatus.OK) {
            log.error("[SSO]调用接口失败：{}, {}", url, response);
            throw new BusinessException("调用SSO服务失败：" + response.getStatusCodeValue());
        }

        log.info("[SSO]接口{}调用成功", url);
        return response.getBody();
    }

    private String obj2json(Object object) {
        if (object == null) {
            return null;
        }
        if (object.getClass().isPrimitive()) {
            return object.toString();
        }
        try {
            return objectMapper.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("对象转json异常", e);
        }
    }

    private URI buildURI(String uri, String... queryParams) {
        Assert.state(StringUtils.hasText(uri), "调用接口路径为空");
        Assert.state(StringUtils.hasText(ssoProperties.getServerAddr()), "SSO服务器地址为空");

        return UriComponentsBuilder.fromHttpUrl(ssoProperties.getServerAddr())
                .path(uri)
                .build(queryParams);
    }
}
