package com.elitescloud.cloudt.authorization.api.provider.cas.support;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.authorization.api.client.AuthenticationClaim;
import com.elitescloud.cloudt.authorization.api.client.common.AuthorizationException;
import com.elitescloud.cloudt.authorization.api.client.config.security.resolver.BearerTokenResolver;
import com.elitescloud.cloudt.authorization.api.client.config.security.resolver.impl.DefaultBearerTokenResolver;
import com.elitescloud.cloudt.authorization.api.provider.cas.CasUserResolver;
import com.elitescloud.cloudt.authorization.api.provider.cas.OidcUserResolver;
import com.elitescloud.cloudt.authorization.api.provider.cas.model.AuthorizeSettingVO;
import com.elitescloud.cloudt.authorization.api.provider.cas.model.OidcUser;
import com.elitescloud.cloudt.authorization.api.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.cloudt.authorization.sdk.cas.model.AuthUserDTO;
import com.elitescloud.cloudt.authorization.sdk.cas.provider.OAuth2ClientProvider;
import com.elitescloud.cloudt.authorization.sdk.cas.provider.UserTransferHelper;
import com.elitescloud.cloudt.authorization.sdk.config.AuthorizationSdkProperties;
import com.elitescloud.cloudt.authorization.sdk.model.OAuthToken;
import com.elitescloud.cloudt.authorization.sdk.model.Result;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.security.authentication.LockedException;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.JwtException;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotBlank;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/19
 */
@Log4j2
public class CasLoginSupportProvider {

    private final AuthorizationSdkProperties sdkProperties;
    private final OAuth2ClientProvider oAuth2ClientProvider;
    private final InternalAuthenticationGranter internalAuthenticationGranter;
    private final UserTransferHelper userTransferHelper;

    private ObjectMapper objectMapper;
    private JwtDecoder jwtDecoder;
    private OidcUserResolver oidcUserResolver;
    private CasUserResolver casUserResolver;
    private final BearerTokenResolver tokenResolver = new DefaultBearerTokenResolver();


    public CasLoginSupportProvider(AuthorizationSdkProperties sdkProperties, OAuth2ClientProvider oAuth2ClientProvider,
                                   InternalAuthenticationGranter internalAuthenticationGranter) {
        this.sdkProperties = sdkProperties;
        this.oAuth2ClientProvider = oAuth2ClientProvider;
        this.internalAuthenticationGranter = internalAuthenticationGranter;
        this.userTransferHelper = UserTransferHelper.getInstance(sdkProperties.getAuthServer());
    }

    /**
     * 获取配置信息
     *
     * @param redirectUrl 重定向路径
     * @return 配置信息
     */
    public ApiResult<AuthorizeSettingVO> getSetting(HttpServletResponse response, @NotBlank String redirectUrl, String state) {
        boolean enabled = ObjectUtil.defaultIfNull(sdkProperties.getCasClient().getEnabled(), false);
        AuthorizeSettingVO settingVO = new AuthorizeSettingVO();
        settingVO.setEnabled(enabled);
        if (enabled) {
            String url = oAuth2ClientProvider.getAuthorizeInfo(response, redirectUrl, state);
            settingVO.setAuthorizeUrl(url);
            settingVO.setAuthServer(sdkProperties.getAuthServer());
        }

        return ApiResult.ok(settingVO);
    }

    /**
     * 授权码换取token
     *
     * @param code 授权码
     * @return token信息
     */
    public ApiResult<OAuthToken> code2Token(HttpServletRequest request, HttpServletResponse response, @NotBlank String code) {
        if (oAuth2ClientProvider == null) {
            return ApiResult.fail("未启用统一身份认证");
        }

        // 授权码换取token
        var tokenResult = oAuth2ClientProvider.code2AccessToken(request, response, code);
        if (!tokenResult.getSuccess()) {
            return ApiResult.fail(tokenResult.getMsg());
        }
        if (tokenResult.getData() == null || !StringUtils.hasText(tokenResult.getData().getIdToken())) {
            return ApiResult.fail("认证异常，未获取到有效token");
        }

        InternalAuthenticationGranter.InternalAuthenticationToken authenticationToken = null;
        if (oidcUserResolver == null) {
            // 默认使用用户账号
            String sub = jwtDecoder.decode(tokenResult.getData().getIdToken()).getSubject();
            authenticationToken = new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.USERNAME, sub);
        } else {
            // 获取认证信息
            var userResult = oAuth2ClientProvider.queryUserInfo(tokenResult.getData().getTokenType(), tokenResult.getData().getAccessToken());
            if (!userResult.getSuccess()) {
                return ApiResult.fail(userResult.getMsg());
            }
            OidcUser oidcUser = null;
            try {
                oidcUser = objectMapper.convertValue(userResult.getData(), OidcUser.class);
            } catch (IllegalArgumentException e) {
                log.error("用户信息转换异常：", e);
                throw new BusinessException("认证失败，请稍后再试");
            }

            // 内部认证，生成token
            authenticationToken = oidcUserResolver.resolve(oidcUser);
        }

        return this.grantToken(request, response, authenticationToken);
    }

    /**
     * token换取token
     * <p>
     * 支持非web端进行统一身份认证
     *
     * @param request 请求
     * @param token   token
     * @return token
     */
    public ApiResult<OAuthToken> token2Token(HttpServletRequest request, HttpServletResponse response, String token) {
        if (CharSequenceUtil.isBlank(token)) {
            token = tokenResolver.resolve(request);
        }
        if (CharSequenceUtil.isBlank(token)) {
            return ApiResult.fail("认证失败，未发现有效token");
        }
        // 解析用户
        var username = this.obtainUsername(token);

        // 认证授权
        InternalAuthenticationGranter.InternalAuthenticationToken authenticationToken = null;
        if (casUserResolver == null) {
            authenticationToken = new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.USERNAME, username);
        } else {
            var userInfo = this.queryUserByUsername(username);
            if (userInfo == null) {
                return ApiResult.fail("认证失败，用户账号不存在");
            }
            var userId = casUserResolver.resolve(userInfo);
            if (userId == null) {
                return ApiResult.fail("认证失败，请确认用户账号存在");
            }
            authenticationToken = new InternalAuthenticationGranter.InternalAuthenticationToken(InternalAuthenticationGranter.IdType.USER_ID, userId.toString());
        }
        return this.grantToken(request, response, authenticationToken);
    }

    private ApiResult<OAuthToken> grantToken(HttpServletRequest request, HttpServletResponse response, InternalAuthenticationGranter.InternalAuthenticationToken authenticationToken) {
        OAuthToken token = null;
        try {
            token = internalAuthenticationGranter.authenticate(request, response, authenticationToken);
        } catch (AuthenticationException e) {
            return ApiResult.fail("认证异常，" + e.getMessage());
        }
        return ApiResult.ok(token);
    }

    private AuthUserDTO queryUserByUsername(@NotBlank String username) {
        Assert.hasText(username, "用户账号为空");
        return userTransferHelper.getUserByUsername(username).getData();
    }

    private String obtainUsername(String token) {
        Jwt jwt = null;
        try {
            jwt = jwtDecoder.decode(token);
        } catch (Exception e) {
            throw new AuthorizationException("认证异常，token不支持");
        }

        return jwt.getClaimAsString(AuthenticationClaim.KEY_USERNAME);
    }

    @Autowired
    public void setObjectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    @Autowired(required = false)
    public void setOidcUserResolver(OidcUserResolver oidcUserResolver) {
        this.oidcUserResolver = oidcUserResolver;
    }

    @Autowired(required = false)
    public void setCasUserResolver(CasUserResolver casUserResolver) {
        this.casUserResolver = casUserResolver;
    }

    @Autowired
    public void setJwtDecoder(JwtDecoder jwtDecoder) {
        this.jwtDecoder = jwtDecoder;
    }
}
