package com.elitescloud.cloudt.authorization.api.provider.config;

import com.elitescloud.cloudt.authorization.api.client.config.support.AuthenticationCallable;
import com.elitescloud.cloudt.authorization.api.client.tool.RedisHelper;
import com.elitescloud.cloudt.authorization.api.provider.cas.controller.CasLoginSupportController;
import com.elitescloud.cloudt.authorization.api.provider.cas.support.CasLoginSupportProvider;
import com.elitescloud.cloudt.authorization.api.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.cloudt.authorization.sdk.cas.AuthorizeCacheable;
import com.elitescloud.cloudt.authorization.sdk.cas.model.AuthorizeDTO;
import com.elitescloud.cloudt.authorization.sdk.cas.provider.OAuth2ClientProvider;
import com.elitescloud.cloudt.authorization.sdk.cas.provider.UserTransferHelper;
import com.elitescloud.cloudt.authorization.sdk.config.AuthorizationSdkProperties;
import com.elitescloud.cloudt.core.config.CloudtCoreAutoConfiguration;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.security.authentication.AccountStatusException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 统一身份认证客户端相关配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/12
 */
@Log4j2
@ConditionalOnClass(CloudtCoreAutoConfiguration.class)
class CloudtCasClientConfig {
    private final AuthorizationSdkProperties sdkProperties;

    public CloudtCasClientConfig(AuthorizationSdkProperties sdkProperties) {
        this.sdkProperties = sdkProperties;
    }

    @Bean
    public CasLoginSupportController oAuth2LoginSupportController(CasLoginSupportProvider supportProvider) {
        return new CasLoginSupportController(supportProvider);
    }

    @Bean
    public CasLoginSupportProvider casLoginSupportProvider(@Autowired(required = false) OAuth2ClientProvider oAuth2ClientProvider,
                                                           @Autowired(required = false) InternalAuthenticationGranter internalAuthenticationGranter) {
        return new CasLoginSupportProvider(sdkProperties, oAuth2ClientProvider, internalAuthenticationGranter);
    }

    @Bean
    public AuthenticationCallable authenticationCallableCasClient(AuthorizationSdkProperties sdkProperties) {
        String authUrl = sdkProperties.getAuthServer();
        return new AuthenticationCallable() {
            @Override
            public void onLoginFailure(HttpServletRequest request, HttpServletResponse response, @Nullable Authentication authentication, @NotNull AuthenticationException exception) {
                var help = UserTransferHelper.getInstance(authUrl);
                if (exception instanceof UsernameNotFoundException || exception instanceof AccountStatusException) {
                    // 出现异常，则禁用账号
                    if (authentication instanceof InternalAuthenticationGranter.InternalAuthenticationToken) {
                        InternalAuthenticationGranter.InternalAuthenticationToken authenticationToken = (InternalAuthenticationGranter.InternalAuthenticationToken) authentication;
                        Long userId = null;
                        if (authenticationToken.getIdType() == InternalAuthenticationGranter.IdType.USER_ID) {
                            userId = Long.parseLong(authenticationToken.getId());
                        } else if (authenticationToken.getIdType() == InternalAuthenticationGranter.IdType.USERNAME) {
                            var user = help.getUserByUsername(authenticationToken.getId()).getData();
                            if (user == null) {
                                log.error("自动禁用账号失败，账号{}不存在", authenticationToken.getId());
                                return;
                            }
                            userId = user.getId();
                        } else if (authenticationToken.getIdType() == InternalAuthenticationGranter.IdType.MOBILE) {
                            var result = help.getUserIdByMobile(List.of(authenticationToken.getId()));
                            if (result.getData() == null || result.getData().isEmpty()) {
                                log.error("自动禁用账号失败，{}", result.getMsg());
                                return;
                            }
                            userId = result.getData().get(authenticationToken.getId());
                        } else if (authenticationToken.getIdType() == InternalAuthenticationGranter.IdType.EMAIL) {
                            var result = help.getUserIdByEmail(List.of(authenticationToken.getId()));
                            if (result.getData() == null || result.getData().isEmpty()) {
                                log.error("自动禁用账号失败，{}", result.getMsg());
                                return;
                            }
                            userId = result.getData().get(authenticationToken.getId());
                        }

                        if (userId != null) {
                            var result = help.updateEnabled(userId, false);
                            if (Boolean.FALSE.equals(result.getSuccess())) {
                                log.error("自动禁用账号失败：" + result.getMsg());
                            }
                            return;
                        }
                        log.error("暂不支持自动禁用账号：" + authenticationToken.getIdType());
                    }
                }
            }
        };
    }

    @Bean
    @ConditionalOnBean(RedisHelper.class)
    public AuthorizeCacheable authorizeCacheableRedis(RedisHelper redisHelper) {
        return new AuthorizeCacheable() {
            private static final String KEY_PREFIX = "cas:authorize:";

            @Override
            public void setCache(String reqId, AuthorizeDTO authorizeDTO) {
                try {
                    redisHelper.execute(redisUtils -> redisUtils.set(KEY_PREFIX + reqId, authorizeDTO, 30, TimeUnit.MINUTES));
                } catch (Exception e) {
                    throw new IllegalStateException("登录异常", e);
                }
            }

            @Override
            public AuthorizeDTO get(String reqId) {
                try {
                    return (AuthorizeDTO) redisHelper.execute(redisUtils -> redisUtils.get(KEY_PREFIX + reqId));
                } catch (Exception e) {
                    throw new IllegalStateException("登录异常", e);
                }
            }
        };
    }
}
