package com.elitescloud.cloudt.authorization.api.provider.provider.rmi.system;

import com.elitescloud.cloudt.authorization.api.client.common.AuthorizationException;
import com.elitescloud.cloudt.authorization.core.SecurityContextUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.provider.SysUserRpcService;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.extern.log4j.Log4j2;
import org.apache.dubbo.config.annotation.DubboReference;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * 系统域的用户信息接口.
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/24
 */
@Log4j2
public class RmiSysUserRpcService {

    private static final String SERVER_ERROR_MSG = "系统域异常";

    @DubboReference
    @Autowired(required = false)
    private SysUserRpcService sysUserRpcService;

    /**
     * 根据用户名查询用户信息
     *
     * @param username 用户名
     * @return 用户信息
     */
    public ApiResult<SysUserDTO> getUserByUsername(String username) {
        try {
            log.info("token：{}", SecurityContextUtil.currentToken());
            return sysUserRpcService.getUserByUsername(username);
        } catch (Exception e) {
            log.error("根据用户名查询用户信息异常", e);
            throw new AuthorizationException(SERVER_ERROR_MSG, e);
        }
    }

    /**
     * 根据微信号查询用户信息
     *
     * @param openid 微信openid
     * @return 用户信息
     */
    public ApiResult<SysUserDTO> getUserByWechatOpenid(String openid) {
        try {
            return sysUserRpcService.getUserByWechatOpenid(openid);
        } catch (Exception e) {
            log.error("根据微信号查询用户信息异常", e);
            throw new AuthorizationException(SERVER_ERROR_MSG, e);
        }
    }

    /**
     * 根据手机号查询用户信息
     *
     * @param mobile 手机号
     * @return 用户信息
     */
    public ApiResult<SysUserDTO> getUserByMobile(String mobile) {
        try {
            return sysUserRpcService.getUserByMobile(mobile);
        } catch (Exception e) {
            log.error("根据手机号查询用户信息异常", e);
            throw new AuthorizationException(SERVER_ERROR_MSG, e);
        }
    }

    /**
     * 根据邮箱查询用户信息
     *
     * @param email 邮箱
     * @return 用户信息
     */
    public ApiResult<SysUserDTO> getUserByEmail(String email) {
        try {
            return sysUserRpcService.getUserByEmail(email);
        } catch (Exception e) {
            log.error("根据邮箱查询用户信息异常", e);
            throw new AuthorizationException(SERVER_ERROR_MSG, e);
        }
    }

    /**
     * 根据用户ID查询用户信息
     *
     * @param userId 用户ID
     * @return 用户信息
     */
    public ApiResult<SysUserDTO> getUserById(String userId) {
        try {
            return sysUserRpcService.getUserById(userId);
        } catch (Exception e) {
            log.error("根据用户ID查询用户信息异常", e);
            throw new AuthorizationException(SERVER_ERROR_MSG, e);
        }
    }

    /**
     * 根据账号查询用户信息
     *
     * @param account 账号
     * @return 用户信息
     */
    public ApiResult<SysUserDTO> getUserByAccount(String account) {
        try {
            return sysUserRpcService.getUserByAccount(account);
        } catch (Exception e) {
            log.error("根据账号查询用户信息异常", e);
            throw new AuthorizationException(SERVER_ERROR_MSG, e);
        }
    }

    /**
     * 微信用户绑定
     *
     * @param userName 用户名
     * @param openid   微信openId
     * @return 绑定结果
     */
    public ApiResult<String> updateUserWechatOpenid(String userName, String openid) {
        try {
            return sysUserRpcService.updateUserWechatOpenid(userName, openid);
        } catch (Exception e) {
            log.error("更新用户微信号查询用户信息异常", e);
            throw new AuthorizationException(SERVER_ERROR_MSG, e);
        }
    }
}
