package com.elitescloud.cloudt.authorization.api.provider.provider.user.support;

import com.elitescloud.cloudt.authorization.api.client.common.AuthorizationException;
import com.elitescloud.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitescloud.cloudt.authorization.api.provider.provider.rmi.system.RmiSysUserRpcService;
import com.elitescloud.cloudt.authorization.api.provider.provider.user.AbstractUserDetailManager;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

import java.util.function.Supplier;

/**
 * 登录用户管理.
 * <p>
 * 调用系统域接口获取用户信息
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/06
 */
@Log4j2
public class SystemUserDetailService extends AbstractUserDetailManager {
    private RmiSysUserRpcService rmiSysUserRpcService;

    public SystemUserDetailService(AuthorizationProperties authorizationProperties) {
        super(authorizationProperties);
        log.info("账号来源：系统中心");
    }

    /**
     * 根据用户名加载用户信息
     *
     * @param username 用户名
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByUsername(String username) {
        return loadUser(() -> rmiSysUserRpcService.getUserByUsername(username).getData());
    }

    /**
     * 根据手机号加载用户信息
     *
     * @param mobile 手机号
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByMobile(String mobile) {
        return loadUser(() -> rmiSysUserRpcService.getUserByMobile(mobile).getData());
    }

    /**
     * 根据邮箱加载用户信息
     *
     * @param email 邮箱
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByEmail(String email) {
        return loadUser(() -> rmiSysUserRpcService.getUserByEmail(email).getData());
    }

    /**
     * 根据用户ID加载用户信息
     *
     * @param userId 用户ID
     * @return 用户信息
     */
    public GeneralUserDetails loadUserById(String userId) {
        return loadUser(() -> rmiSysUserRpcService.getUserById(userId).getData());
    }

    /**
     * 根据账号加载用户信息
     *
     * @param account 账号
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByAccount(String account) {
        return loadUser(() -> rmiSysUserRpcService.getUserByAccount(account).getData());
    }

    /**
     * 根据用户openid加载用户信息
     *
     * @param wechatOpenid 微信openId
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByWechatOpenid(String wechatOpenid) {
        return loadUser(() -> rmiSysUserRpcService.getUserByWechatOpenid(wechatOpenid).getData());
    }

    private GeneralUserDetails loadUser(Supplier<SysUserDTO> userDTOSupplier) {
        // 查询用户信息
        SysUserDTO user = userDTOSupplier.get();
        if (user == null) {
            return null;
        }

        return super.wrapUserDetails(user);
    }

    @Autowired
    public void setRmiSysUserRpcService(RmiSysUserRpcService rmiSysUserRpcService) {
        this.rmiSysUserRpcService = rmiSysUserRpcService;
    }
}
