package com.elitescloud.cloudt.authorization.api.provider.provider.wechat;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.authorization.api.client.tool.RedisHelper;
import com.elitescloud.cloudt.authorization.api.provider.provider.wechat.param.BaseWechatResult;
import com.elitescloud.cloudt.authorization.api.provider.provider.wechat.param.WechatPhoneInfo;
import com.elitescloud.cloudt.common.exception.BusinessException;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.function.Function;
import java.util.function.UnaryOperator;

/**
 * 微信工具.
 *
 * @author Kaiser（wang shao）
 * @date 2/14/2023
 */
@Log4j2
public class WechatTemplate {

    private final Function<String, String> getAppSecretFunction;
    private final RedisHelper redisHelper;
    private final WechatTool wechatTool;

    public WechatTemplate(@NotNull UnaryOperator<String> getAppSecretFunction,
                          @NotNull RedisHelper redisHelper) {
        this.getAppSecretFunction = getAppSecretFunction;
        this.redisHelper = redisHelper;
        this.wechatTool = WechatTool.getInstance();
    }

    /**
     * 获取手机号
     *
     * @param appId 应用ID
     * @param code  授权码
     * @return 手机号信息
     */
    public WechatPhoneInfo getPhoneInfo(@NotBlank String appId, @NotBlank String code) {
        Assert.hasText(appId, "微信应用ID为空");
        Assert.hasText(code, "微信授权码为空");

        return this.executeWithToken(appId, token -> wechatTool.getPhoneNumber(token, code));
    }

    private <T extends BaseWechatResult> T executeWithToken(@NotBlank String appId, @NotNull Function<String, T> supplier) {
        // 获取token
        String accessToken = getAccessToken(appId);
        T result = supplier.apply(accessToken);
        if (result.isSuccess()) {
            return result;
        }

        // 刷新token重试
        clearAccessToken(appId);
        result = supplier.apply(accessToken);
        if (!result.isSuccess()) {
            log.error("调用微信接口失败：{}", result);
        }

        return result;
    }

    private String getAccessToken(String appId) {
        // 先从缓存获取
        String key = "wechat:accesstoken:" + appId;
        String accessToken = (String) redisHelper.execute(redisUtils -> redisUtils.get(key));
        if (StringUtils.hasText(accessToken)) {
            return accessToken;
        }

        // 调用微信接口生成token
        String appSecret = getAppSecretFunction.apply(appId);
        var tokenResult = wechatTool.getAccessToken(appId, appSecret);
        if (tokenResult == null || CharSequenceUtil.isBlank(tokenResult.getAccessToken())) {
            throw new BusinessException("微信授权异常，请稍后再试");
        }

        // 将token缓存
        redisHelper.execute(redisUtils -> redisUtils.set(key, tokenResult.getAccessToken(), tokenResult.getExpiresIn()));
        return tokenResult.getAccessToken();
    }

    private void clearAccessToken(String appId) {
        String key = "wechat:accesstoken:" + appId;
        redisHelper.execute(redisUtils -> {
            redisUtils.del(key);
            return null;
        });
    }
}
