package com.elitescloud.cloudt.authorization.api.provider.provider.wechat;

import com.elitescloud.cloudt.authorization.api.provider.provider.wechat.param.WechatAccessToken;
import com.elitescloud.cloudt.authorization.api.provider.provider.wechat.param.WechatPhoneInfo;
import com.elitescloud.cloudt.authorization.api.provider.provider.wechat.param.WechatToken;
import com.elitescloud.cloudt.authorization.api.provider.provider.wechat.param.WechatUserInfo;
import com.elitescloud.cloudt.authorization.sdk.util.RestTemplateFactory;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.log4j.Log4j2;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.NonNull;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.validation.constraints.NotBlank;
import java.util.Map;

/**
 * 微信工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/20
 */
@Log4j2
class WechatTool {

    private final RestTemplate restTemplate;
    private final ObjectMapper objectMapper;

    private WechatTool() {
        this.restTemplate = RestTemplateFactory.instance();
        this.objectMapper = new ObjectMapper();
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /**
     * 获取微信工具类
     *
     * @return
     */
    public static WechatTool getInstance() {
        return new WechatTool();
    }

    /**
     * 获取accessToken
     * <p>
     * 接口文档<a href="https://developers.weixin.qq.com/doc/offiaccount/Basic_Information/Get_access_token.html">地址</a>
     *
     * @param appId     应用ID
     * @param appSecret 应用密码
     * @return token
     */
    public WechatToken getAccessToken(@NotBlank String appId, @NotBlank String appSecret) {
        String url = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=APPID&secret=APPSECRET";

        url = url.replace("APPID", appId)
                .replace("APPSECRET", appSecret);
        String resp = this.execGet(url);
        if (StringUtils.hasText(resp)) {
            log.info("微信获取AccessToken：{}", resp);
            try {
                return objectMapper.readValue(resp, new TypeReference<>() {
                });
            } catch (JsonProcessingException e) {
                log.error("获取微信access_token异常：", e);
                throw new BusinessException("请求微信服务获取access_token异常", e);
            }
        }
        return null;
    }

    /**
     * 获取手机号
     * <p>
     * 接口文档<a href="https://developers.weixin.qq.com/miniprogram/dev/OpenApiDoc/user-info/phone-number/getPhoneNumber.html">地址</a>
     *
     * @param accessToken token
     * @param code        授权码
     * @return token
     */
    public WechatPhoneInfo getPhoneNumber(@NotBlank String accessToken, @NotBlank String code) {
        String url = "https://api.weixin.qq.com/wxa/business/getuserphonenumber?access_token=ACCESS_TOKEN";

        url = url.replace("ACCESS_TOKEN", accessToken);
        String resp = this.execPost(url, Map.of("code", code));
        if (StringUtils.hasText(resp)) {
            log.info("微信获取手机号：{}", resp);
            try {
                return objectMapper.readValue(resp, new TypeReference<>() {
                });
            } catch (JsonProcessingException e) {
                log.error("获取微信手机号异常：", e);
                throw new BusinessException("请求微信服务获取手机号异常", e);
            }
        }
        return null;
    }

    /**
     * OAuth2 code换 accessToken
     *
     * @param appId
     * @param secret
     * @param code
     * @return
     */
    public WechatAccessToken oauth2AccessToken(@NonNull String appId, @NonNull String secret, @NonNull String code) {
        String url = "https://api.weixin.qq.com/sns/oauth2/access_token?appid=APPID&secret=SECRET&code=CODE&grant_type=authorization_code";

        url = url.replace("APPID", appId)
                .replace("SECRET", secret)
                .replace("CODE", code)
        ;
        String resp = execGet(url);
        if (StringUtils.hasText(resp)) {
            try {
                return objectMapper.readValue(resp, new TypeReference<>() {
                });
            } catch (JsonProcessingException e) {
                log.error("转换微信access_token异常：", e);
                throw new BusinessException("请求微信服务获取access_token异常", e);
            }
        }
        return null;
    }

    /**
     * 获取微信用户信息
     *
     * @param accessToken
     * @param openId
     * @return
     */
    public WechatUserInfo snsUserInfo(String accessToken, String openId) {
        String url = "https://api.weixin.qq.com/sns/userinfo?access_token=ACCESS_TOKEN&openid=OPENID&lang=zh_CN";

        url = url.replace("ACCESS_TOKEN", accessToken)
                .replace("OPENID", openId)
        ;
        String resp = execGet(url);
        if (StringUtils.hasText(resp)) {
            try {
                return objectMapper.readValue(resp, new TypeReference<>() {
                });
            } catch (JsonProcessingException e) {
                log.error("转换微信用户信息异常：", e);
                throw new BusinessException("请求微信服务获取用户信息异常", e);
            }
        }
        return null;
    }

    private String execGet(String url) {
        log.info("请求微信接口：{}", url);
        try {
            ResponseEntity<String> resp = restTemplate.exchange(url, HttpMethod.GET, null, new ParameterizedTypeReference<>() {
            });
            log.info("请求微信接口返回结果：{}, {}", resp.getStatusCode(), resp.getBody());
            if (resp.getStatusCode().is2xxSuccessful()) {
                return resp.getBody();
            }
        } catch (RestClientException e) {
            log.error("请求微信接口异常：", e);
            throw new BusinessException("请求微信接口异常", e);
        }

        return null;
    }

    private String execPost(String url, Object entity) {
        log.info("请求微信接口：{}", url);
        try {
            ResponseEntity<String> resp = restTemplate.exchange(url, HttpMethod.POST, new HttpEntity<>(entity), new ParameterizedTypeReference<>() {
            });
            log.info("请求微信接口返回结果：{}, {}", resp.getStatusCode(), resp.getBody());
            if (resp.getStatusCode().is2xxSuccessful()) {
                return resp.getBody();
            }
        } catch (RestClientException e) {
            log.error("请求微信接口异常：", e);
            throw new BusinessException("请求微信接口异常", e);
        }

        return null;
    }
}
