package com.elitescloud.cloudt.authorization.api.provider.security.impl;

import com.elitescloud.cloudt.authorization.api.client.common.AuthorizationException;
import com.elitescloud.cloudt.authorization.api.client.tool.RedisHelper;
import com.elitescloud.cloudt.authorization.api.provider.config.servlet.oauth2.OAuth2AuthorizationCodeRequestCache;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.oauth2.server.authorization.authentication.OAuth2AuthorizationCodeRequestAuthenticationToken;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

/**
 * 认证请求缓存.
 * <p>
 * 针对不支持重定向，依靠state维持认证状态的客户端，缓存用户认证前的授权码请求信息
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/5
 */
@Log4j2
public class RedisOAuth2AuthorizationCodeRequestCache implements OAuth2AuthorizationCodeRequestCache {

    private final RedisHelper redisHelper;

    public RedisOAuth2AuthorizationCodeRequestCache(RedisHelper redisHelper) {
        this.redisHelper = redisHelper;
    }

    @Override
    public void setAuthenticationToken(String reqId, OAuth2AuthorizationCodeRequestAuthenticationToken authenticationToken, Duration timeToLive) {
        try {
            redisHelper.execute(redisUtils -> {
                if (timeToLive == null) {
                    redisUtils.set(reqId, authenticationToken);
                    return null;
                }

                redisUtils.set(reqId, authenticationToken, timeToLive.toSeconds(), TimeUnit.SECONDS);
                return null;
            });
        } catch (Exception e) {
            log.error("缓存授权码认证请求失败", e);
            throw new AuthorizationException("授权码认证失败", e);
        }
    }

    @Override
    public OAuth2AuthorizationCodeRequestAuthenticationToken getAuthenticationToken(String reqId) {
        try {
            return (OAuth2AuthorizationCodeRequestAuthenticationToken) redisHelper.execute(redisUtils -> redisUtils.get(reqId));
        } catch (Exception e) {
            log.error("获取缓存授权码认证请求失败", e);
            throw new AuthorizationException("授权码认证失败", e);
        }
    }

    @Override
    public void removeAuthenticationToken(String reqId) {
        try {
            redisHelper.execute(redisUtils -> {
                redisUtils.del(reqId);
                return null;
            });
        } catch (Exception e) {
            log.error("删除缓存授权码认证请求失败", e);
        }
    }
}
