package com.elitescloud.cloudt.authorization.api.provider.security.impl;

import com.elitescloud.cloudt.authorization.api.client.common.AuthorizationException;
import com.elitescloud.cloudt.authorization.api.client.common.LoginType;
import com.elitescloud.cloudt.authorization.api.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.cloudt.authorization.api.provider.config.system.TenantProperties;
import com.elitescloud.cloudt.authorization.api.provider.security.AuthenticationCheckService;
import com.elitescloud.cloudt.common.constant.Terminal;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.core.tenant.support.TenantClientProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.core.AuthenticationException;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;

/**
 * 用户的租户信息检查.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/20
 */
@Log4j2
public class TenantAuthenticationCheckServiceImpl<T extends AbstractCustomAuthenticationToken<T>> implements AuthenticationCheckService<T> {

    private final TenantProperties tenantProperties;
    private final TenantClientProvider tenantClientProvider;

    public TenantAuthenticationCheckServiceImpl(TenantProperties tenantProperties, TenantClientProvider tenantClientProvider) {
        this.tenantProperties = tenantProperties;
        this.tenantClientProvider = tenantClientProvider;
    }

    @Override
    public void additionalAuthenticationChecks(GeneralUserDetails user, T authentication) throws AuthenticationException {
        boolean limitLogin = Boolean.TRUE.equals(tenantProperties.isLimitTenantLogin());

        if (!tenantClientProvider.enabledTenant() || !limitLogin) {
            // 未启用租户，直接返回
            return;
        }

        if (user.isOperation()) {
            // 运营机构无限制
            return;
        }

        if (authentication.loginType() == LoginType.INTERNAL) {
            // 内部认证
            return;
        }

        // 判断请求域名所属租户
        var requestTenant = tenantClientProvider.obtainTenantFromRequest();
        if (requestTenant != null) {
            if (user.getTenant() != null && user.getTenant().getId().longValue() == requestTenant.getId()) {
                // 匹配上
                return;
            }
            throw new AuthorizationException("请转至对应租户下的站点登录");
        }

        // 判断是否默认站点
        if (!tenantClientProvider.isDefaultDomainRequest()) {
            throw new AuthorizationException("未知当前站点所属租户");
        }
        printDomain();

        // 管理端，校验限制
        if (Terminal.BACKEND.equals(authentication.getTerminal())) {
            throw new AuthorizationException("请转至对应租户下的站点登录");
        }
    }

    private void printDomain() {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if (requestAttributes == null) {
            return;
        }

        HttpServletRequest request = ((ServletRequestAttributes) requestAttributes).getRequest();
        String uri = request.getRequestURL().toString();
        String domain = HttpServletUtil.obtainDomain(request);
        log.info("请求路径：{}, {}", uri, domain);
    }
}
