package com.elitescloud.cloudt.authorization.sdk.cas.provider;

import com.elitescloud.cloudt.authorization.sdk.cas.model.thirdparty.WechatMiniProgramDTO;
import com.elitescloud.cloudt.authorization.sdk.cas.model.thirdparty.WechatServiceDTO;
import com.elitescloud.cloudt.authorization.sdk.cas.model.thirdparty.WechatSubscriptionDTO;
import com.elitescloud.cloudt.authorization.sdk.model.Result;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.util.Assert;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;

/**
 * 第三方账号同步助手.
 *
 * @author Kaiser（wang shao）
 * @date 2/14/2023
 */
public final class ThirdPartyAccountTransferHelper extends BaseTransferHelper {
    private static final Logger LOG = LoggerFactory.getLogger(ThirdPartyAccountTransferHelper.class);
    private static ThirdPartyAccountTransferHelper instance = null;

    private ThirdPartyAccountTransferHelper(String authServer) {
        super(authServer);
    }

    /**
     * 创建工具实例
     *
     * @param authServer 认证服务器地址
     * @return 工具实例
     */
    public static ThirdPartyAccountTransferHelper getInstance(@NotBlank String authServer) {
        synchronized (ThirdPartyAccountTransferHelper.class) {
            if (instance == null) {
                instance = new ThirdPartyAccountTransferHelper(authServer);
            }
        }
        return instance;
    }

    /**
     * 同步微信服务号
     *
     * @param dto 服务号信息
     * @return 账号ID
     */
    public Result<Long> upsertWechatService(@NotNull WechatServiceDTO dto) {
        Assert.notNull(dto, "微信服务号信息为空");
        LOG.info("同步微信服务号：{}", dto.getAppId());

        return remoteExchange(CasUrlConstant.URI_THIRD_PARTY_UPSERT_WECHAT_SERVICE, HttpMethod.POST, new HttpEntity<>(dto), new ParameterizedTypeReference<>() {
        });
    }

    /**
     * 同步微信小程序
     *
     * @param dto 小程序信息
     * @return 账号ID
     */
    public Result<Long> upsertWechatMiniProgram(@NotNull WechatMiniProgramDTO dto) {
        Assert.notNull(dto, "微信小程序信息为空");
        LOG.info("同步微信小程序：{}", dto.getAppId());

        return remoteExchange(CasUrlConstant.URI_THIRD_PARTY_UPSERT_WECHAT_MINI, HttpMethod.POST, new HttpEntity<>(dto), new ParameterizedTypeReference<>() {
        });
    }

    /**
     * 同步微信订阅号
     *
     * @param dto 订阅号信息
     * @return 账号ID
     */
    public Result<Long> upsertWechatSubscription(@NotNull WechatSubscriptionDTO dto) {
        Assert.notNull(dto, "微信订阅号信息为空");
        LOG.info("同步微信订阅号：{}", dto.getAppId());

        return remoteExchange(CasUrlConstant.URI_THIRD_PARTY_UPSERT_WECHAT_SUBSCRIPTION, HttpMethod.POST, new HttpEntity<>(dto), new ParameterizedTypeReference<>() {
        });
    }

    /**
     * 更新账号状态
     *
     * @param accountId 账号ID
     * @param enabled   启用状态
     * @return 账号ID
     */
    public Result<Long> updateEnabled(long accountId, boolean enabled) {
        LOG.info("同步第三方账号状态：{}, {}", accountId, enabled);

        return remoteExchange(CasUrlConstant.URI_THIRD_PARTY_UPDATE_ENABLED, HttpMethod.PATCH, null,
                new ParameterizedTypeReference<>() {
                }, accountId, enabled);
    }

    /**
     * 删除账号
     *
     * @param accountId 账号ID
     * @return 账号ID
     */
    public Result<Long> delete(long accountId) {
        LOG.info("同步删除第三方账号状态：{}", accountId);

        return remoteExchange(CasUrlConstant.URI_THIRD_PARTY_DELETE, HttpMethod.DELETE, null,
                new ParameterizedTypeReference<>() {
                }, accountId);
    }
}
