package com.elitescloud.cloudt.authorization.sdk.sso.configurer;

import com.elitescloud.cloudt.authorization.sdk.config.AuthorizationSdkProperties;
import com.elitescloud.cloudt.authorization.sdk.sso.configurer.filter.CloudtSsoFilter;
import com.elitescloud.cloudt.authorization.sdk.sso.SsoProvider;
import com.elitescloud.cloudt.authorization.sdk.sso.TicketResolver;
import com.elitescloud.cloudt.authorization.sdk.sso.impl.CookieTicketResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.config.annotation.web.HttpSecurityBuilder;
import org.springframework.security.config.annotation.web.configurers.AbstractHttpConfigurer;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.preauth.AbstractPreAuthenticatedProcessingFilter;

import javax.servlet.Filter;

/**
 * 单点登录过滤器配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/14
 */
public class SsoFilterConfigurer<H extends HttpSecurityBuilder<H>>
        extends AbstractHttpConfigurer<SsoFilterConfigurer<H>, H> {
    private static final Logger log = LoggerFactory.getLogger(SsoFilterConfigurer.class);

    private final AuthorizationSdkProperties sdkProperties;

    /**
     * filter要添加的类之前
     */
    private Class<? extends Filter> addFilterBeforeClass;
    private SsoProvider ssoProvider;
    private TicketResolver ticketResolver;
    private AuthenticationSuccessHandler authenticationSuccessHandler;
    private AuthenticationFailureHandler authenticationFailureHandler;

    public SsoFilterConfigurer(AuthorizationSdkProperties sdkProperties) {
        this.sdkProperties = sdkProperties;
    }

    @Override
    public void init(H builder) throws Exception {
        if (ticketResolver == null) {
            this.ticketResolver = new CookieTicketResolver(sdkProperties);
        }
        super.init(builder);
    }

    @Override
    public void configure(H builder) throws Exception {
        if (!sdkProperties.getSso().getEnabled()) {
            // 未启用
            return;
        }

        // 添加filter
        CloudtSsoFilter filter = new CloudtSsoFilter(sdkProperties);
        if (ssoProvider == null) {
            log.warn("SsoProvider 为空");
        }
        filter.setSsoProvider(ssoProvider);
        filter.setTicketResolver(ticketResolver);
        filter.setAuthenticationSuccessHandler(authenticationSuccessHandler);
        if (addFilterBeforeClass == null) {
            addFilterBeforeClass = AbstractPreAuthenticatedProcessingFilter.class;
        }
        builder.addFilterBefore(filter, addFilterBeforeClass);

        super.configure(builder);
    }

    public SsoFilterConfigurer<H> setAddFilterBeforeClass(Class<? extends Filter> clazz) {
        this.addFilterBeforeClass = clazz;
        return this;
    }

    public SsoFilterConfigurer<H> setSsoProvider(SsoProvider ssoProvider) {
        this.ssoProvider = ssoProvider;
        return this;
    }

    public SsoFilterConfigurer<H> setTicketResolver(TicketResolver ticketResolver) {
        this.ticketResolver = ticketResolver;
        return this;
    }

    public SsoFilterConfigurer<H> setAuthenticationSuccessHandler(AuthenticationSuccessHandler authenticationSuccessHandler) {
        this.authenticationSuccessHandler = authenticationSuccessHandler;
        return this;
    }

    public SsoFilterConfigurer<H> setAuthenticationFailureHandler(AuthenticationFailureHandler authenticationFailureHandler) {
        this.authenticationFailureHandler = authenticationFailureHandler;
        return this;
    }
}
