package com.elitescloud.cloudt.authorization.sdk.util;

import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.boot.web.client.RestTemplateCustomizer;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.client.RestTemplate;

import java.time.Duration;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * RestTemplate.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/14
 */
public class RestTemplateFactory {

    private RestTemplateFactory() {
    }

    /**
     * 创建RestTemplate实例
     *
     * @return restTemplate
     */
    public static RestTemplate instance() {
        return instance(null);
    }

    /**
     * 创建RestTemplate实例
     *
     * @return restTemplate
     */
    public static RestTemplate instance(Function<RestTemplateBuilder, RestTemplateBuilder> restTemplateBuilderFunction) {
        var builder = new RestTemplateBuilder()
                .requestFactory(RestTemplateFactory::getClientHttpRequestFactory)
                .customizers(restTemplateCustomizer());
        if (restTemplateBuilderFunction != null) {
            builder = restTemplateBuilderFunction.apply(builder);
        }
        return builder.build();
    }

    private static ClientHttpRequestFactory getClientHttpRequestFactory() {
        SSLConnectionSocketFactory sslConnectionSocketFactory = null;

        try {
            SSLContextBuilder sslContextBuilder = new SSLContextBuilder();
            sslContextBuilder.loadTrustMaterial(null, new TrustSelfSignedStrategy());
            sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContextBuilder.build(), NoopHostnameVerifier.INSTANCE);
        } catch (Exception e) {
            e.printStackTrace();
        }

        HttpClientBuilder httpClientBuilder = HttpClients.custom();
        httpClientBuilder.setSSLSocketFactory(sslConnectionSocketFactory);

        var requestFactory = new HttpComponentsClientHttpRequestFactory();
        requestFactory.setHttpClient(httpClientBuilder.build());
        requestFactory.setConnectTimeout((int) Duration.ofSeconds(10).toMillis());
        requestFactory.setReadTimeout((int) Duration.ofSeconds(60).toMillis());
        return requestFactory;
    }

    private static RestTemplateCustomizer restTemplateCustomizer() {
        var objectMapper = ObjectMapperFactory.instance();

        return template -> {
            for (var convert : template.getMessageConverters()) {
                if (convert instanceof MappingJackson2HttpMessageConverter) {
                    ((MappingJackson2HttpMessageConverter) convert).setObjectMapper(objectMapper);
                    return;
                }
            }
            template.getMessageConverters().add(new MappingJackson2HttpMessageConverter(objectMapper));
        };
    }
}
