package com.elitescloud.cloudt.authorization.sdk.cas.task;

import com.elitescloud.cloudt.authorization.sdk.cas.UserSourceProvider;
import com.elitescloud.cloudt.authorization.sdk.cas.provider.UserTransferHelper;
import com.elitescloud.cloudt.authorization.sdk.config.AuthorizationSdkProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

/**
 * 自动同步用户任务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/15
 */
public class AutoSyncUserTask implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(AutoSyncUserTask.class);
    private static final int MAX_SIZE = 100;

    private final UserSourceProvider userSyncProvider;
    private final UserTransferHelper transferHelper;
    /**
     * 每次查询数据量
     */
    private int syncSize = 50;
    /**
     * 是否一批失败时全部回滚
     */
    private boolean rollbackAllOnException;

    public AutoSyncUserTask(String authServer, UserSourceProvider userSyncProvider) {
        this.userSyncProvider = userSyncProvider;
        this.transferHelper = UserTransferHelper.getInstance(authServer);
    }

    @Override
    public void run() {
        if (userSyncProvider == null) {
            return;
        }

        if (syncSize < 1) {
            // 小于1，则不再同步
            return;
        }
        int size = Math.min(syncSize, MAX_SIZE);

        // 查询需要同步的用户
        var users = userSyncProvider.queryUser(size);
        while (!CollectionUtils.isEmpty(users)) {
            // 调用服务端接口同步用户
            var result = transferHelper.upsertUserBatch(rollbackAllOnException, users);
            if (!result.getSuccess()) {
                // 失败则终止
                LOG.error("CAS同步用户账号失败：{}", result.getMsg());
                break;
            }

            // 成功后回调，更新本地记录
            if (!CollectionUtils.isEmpty(result.getData())) {
                userSyncProvider.successCallback(result.getData());
            }

            users = userSyncProvider.queryUser(size);
        }
    }

    public void setSyncSize(int syncSize) {
        Assert.isTrue(syncSize < MAX_SIZE, "最大值为" + MAX_SIZE);
        this.syncSize = syncSize;
    }

    public void setRollbackAllOnException(boolean rollbackAllOnException) {
        this.rollbackAllOnException = rollbackAllOnException;
    }
}
