package com.elitescloud.cloudt.authorization.sdk.cas.config;

import com.elitescloud.cloudt.authorization.sdk.cas.AuthorizeCacheable;
import com.elitescloud.cloudt.authorization.sdk.cas.UserSourceProvider;
import com.elitescloud.cloudt.authorization.sdk.cas.provider.OAuth2ClientProvider;
import com.elitescloud.cloudt.authorization.sdk.cas.task.CasClientScheduler;
import com.elitescloud.cloudt.authorization.sdk.config.AuthorizationSdkProperties;
import com.elitescloud.cloudt.authorization.sdk.resolver.UniqueRequestResolver;
import org.slf4j.Logger;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.lang.NonNull;

import static org.slf4j.LoggerFactory.getLogger;

/**
 * 统一身份认证客户端配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/11
 */
@ConditionalOnProperty(prefix = AuthorizationSdkProperties.CONFIG_PREFIX + ".cas-client", name = "enabled", havingValue = "true")
public class CasClientConfig implements ApplicationContextAware, InitializingBean {
    private static final Logger LOG = getLogger(CasClientConfig.class);
    private final AuthorizationSdkProperties sdkProperties;
    private ApplicationContext applicationContext;

    public CasClientConfig(AuthorizationSdkProperties sdkProperties) {
        this.sdkProperties = sdkProperties;
        LOG.info("统一身份认证客户端认证启用");
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        var userSourceProvider = applicationContext.getBeanProvider(UserSourceProvider.class).getIfAvailable();
        if (userSourceProvider == null) {
            LOG.warn("未能成功同步用户账号，未发现配置有效的{}", UserSourceProvider.class.getName());
        }
    }

    @Override
    public void setApplicationContext(@NonNull ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Bean
    @ConditionalOnMissingBean
    OAuth2ClientProvider oAuth2ClientProvider(ObjectProvider<AuthorizeCacheable> authorizeCacheable) {
        return new OAuth2ClientProvider(sdkProperties, authorizeCacheable.getIfAvailable());
    }

    @Bean
    CasClientScheduler authorizationSdkScheduler() {
        return new CasClientScheduler(sdkProperties);
    }
}
