package com.elitescloud.cloudt.authorization.sdk.cas.provider;

import com.elitescloud.cloudt.authorization.sdk.cas.model.AuthUserDTO;
import com.elitescloud.cloudt.authorization.sdk.model.Result;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.util.Assert;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户传输服务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/16
 */
public final class UserTransferHelper extends BaseTransferHelper {
    private static final Logger LOG = LoggerFactory.getLogger(UserTransferHelper.class);

    private static UserTransferHelper transferHelper = null;

    private UserTransferHelper(String authServer) {
        super(authServer);
    }

    /**
     * 创建工具实例
     *
     * @param authServer 认证服务器地址
     * @return
     */
    public static UserTransferHelper getInstance(@NotBlank String authServer) {
        synchronized (UserTransferHelper.class) {
            if (transferHelper == null) {
                transferHelper = new UserTransferHelper(authServer);
            }
        }
        return transferHelper;
    }

    /**
     * 同步用户
     *
     * @param userInfo 用户信息
     * @return 用户ID
     */
    public Result<Long> upsertUser(@NotNull AuthUserDTO userInfo) {
        Assert.notNull(userInfo, "用户信息为空");
        LOG.info("向统一身份认证中心同步账号：{}", userInfo.getUsername());

        return remoteExchange(CasUrlConstant.URI_USER_UPSERT, HttpMethod.POST, new HttpEntity<>(userInfo),
                new ParameterizedTypeReference<>() {
                });
    }

    /**
     * 批量同步用户
     *
     * @param rollBackAllOnException 出现异常，是否回滚所有
     * @param userInfoList           用户信息列表
     * @return username map userId
     */
    public Result<HashMap<String, Long>> upsertUserBatch(boolean rollBackAllOnException, @NotEmpty List<AuthUserDTO> userInfoList) {
        Assert.notEmpty(userInfoList, "用户信息为空");
        LOG.info("向统一身份认证中心同步账号：{}", userInfoList.stream().map(AuthUserDTO::getUsername).collect(Collectors.toList()));

        return remoteExchange(CasUrlConstant.URI_USER_UPSERT_BATCH, HttpMethod.POST, new HttpEntity<>(userInfoList),
                new ParameterizedTypeReference<>() {
                }, rollBackAllOnException);
    }

    /**
     * 更新用户状态
     *
     * @param userId  用户ID
     * @param enabled 启用状态
     * @return 用户ID
     */
    public Result<Long> updateEnabled(long userId, boolean enabled) {
        LOG.info("向统一身份认证中心同步账号状态：{}, {}", userId, enabled);

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_ENABLED, HttpMethod.PATCH, null,
                new ParameterizedTypeReference<>() {
                }, userId, enabled);
    }

    /**
     * 修改用户的密码
     *
     * @param userId 用户账号ID
     * @param pd     密码，明文
     * @return 用户ID
     */
    public Result<Long> updatePassword(long userId, @NotBlank String pd) {
        Assert.hasText(pd, "密码为空");
        LOG.info("向统一身份认证中心同步账号密码：{}, {}", userId, pd);

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_PWD, HttpMethod.PATCH, null,
                new ParameterizedTypeReference<>() {
                }, userId, pd);
    }

    /**
     * 修改用户的账号到期时间
     *
     * @param userId      用户账号ID
     * @param expiredTime 账号过期时间，格式yyyy-MM-dd HH:mm:ss
     * @return 用户ID
     */
    public Result<Long> updateExpiredTime(long userId, @NotBlank String expiredTime) {
        Assert.hasText(expiredTime, "过期时间为空");
        LOG.info("向统一身份认证中心同步账号过期时间：{}, {}", userId, expiredTime);

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_EXPIRED_TIME, HttpMethod.PATCH, null,
                new ParameterizedTypeReference<>() {
                }, userId, expiredTime);
    }


    /**
     * 修改用户的密码到期时间
     *
     * @param userId      用户账号ID
     * @param expiredTime 密码过期时间，格式yyyy-MM-dd HH:mm:ss
     * @return 用户ID
     */
    public Result<Long> updatePwdExpiredTime(long userId, String expiredTime) {
        Assert.hasText(expiredTime, "过期时间为空");
        LOG.info("向统一身份认证中心同步密码过期时间：{}, {}", userId, expiredTime);

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_PWD_EXPIRED_TIME, HttpMethod.PATCH, null,
                new ParameterizedTypeReference<>() {
                }, userId, expiredTime);
    }

    /**
     * 删除用户账号
     *
     * @param userId 用户账号ID
     * @return 用户账号ID
     */
    public Result<Long> delete(long userId) {
        LOG.info("向统一身份认证中心删除账号：{}", userId);

        return remoteExchange(CasUrlConstant.URI_USER_DELETE, HttpMethod.DELETE, null,
                new ParameterizedTypeReference<>() {
                }, userId);
    }

    /**
     * 根据ID获取用户信息
     *
     * @param userId 用户账号ID
     * @return 用户信息
     */
    public Result<AuthUserDTO> getUser(long userId) {
        return remoteExchange(CasUrlConstant.URI_USER_GET, HttpMethod.GET, null,
                new ParameterizedTypeReference<>() {
                }, userId);
    }

    /**
     * 根据username获取用户信息
     *
     * @param username 用户账号
     * @return 用户信息
     */
    public Result<AuthUserDTO> getUserByUsername(@NotBlank String username) {
        return remoteExchange(CasUrlConstant.URI_USER_GET_BY_USERNAME, HttpMethod.GET, null,
                new ParameterizedTypeReference<>() {
                }, username);
    }

    /**
     * 获取用户列表
     *
     * @param userIds 用户账号ID
     * @return 用户信息
     */
    public Result<ArrayList<AuthUserDTO>> getUserList(@NotEmpty List<Long> userIds) {
        Assert.notEmpty(userIds, "用户信息为空");

        return remoteExchange(CasUrlConstant.URI_USER_LIST, HttpMethod.POST, new HttpEntity<>(userIds),
                new ParameterizedTypeReference<>() {
                });
    }

    /**
     * 获取用户ID
     *
     * @param usernames 用户账号
     * @return 用户ID
     */
    public Result<HashMap<String, Long>> getUserIdByUsername(@NotEmpty List<String> usernames) {
        Assert.notEmpty(usernames, "用户账号为空");

        return remoteExchange(CasUrlConstant.URI_USER_ID_BY_USERNAME, HttpMethod.POST, new HttpEntity<>(usernames),
                new ParameterizedTypeReference<>() {
                });
    }

    /**
     * 获取用户ID
     *
     * @param mobiles 手机号
     * @return 用户ID
     */
    public Result<HashMap<String, Long>> getUserIdByMobile(@NotEmpty List<String> mobiles) {
        Assert.notEmpty(mobiles, "用户账号为空");

        return remoteExchange(CasUrlConstant.URI_USER_ID_BY_MOBILE, HttpMethod.POST, new HttpEntity<>(mobiles),
                new ParameterizedTypeReference<>() {
                });
    }

    /**
     * 获取用户ID
     *
     * @param emails 邮箱
     * @return 用户ID
     */
    public Result<HashMap<String, Long>> getUserIdByEmail(@NotEmpty List<String> emails) {
        Assert.notEmpty(emails, "邮箱为空");

        return remoteExchange(CasUrlConstant.URI_USER_ID_BY_EMAIL, HttpMethod.POST, new HttpEntity<>(emails),
                new ParameterizedTypeReference<>() {
                });
    }
}
