package com.elitescloud.boot.auth.cas.provider;

import com.elitescloud.boot.auth.cas.common.PwdRecordTypeEnum;
import com.elitescloud.boot.auth.cas.model.AuthClientUserDTO;
import com.elitescloud.boot.auth.cas.model.AuthUserDTO;
import com.elitescloud.boot.auth.cas.model.AuthUserQueryDTO;
import com.elitescloud.boot.auth.cas.model.pwd.AuthPwdUpdateDTO;
import com.elitescloud.boot.auth.model.Result;
import com.fasterxml.jackson.core.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.Assert;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户传输服务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/16
 */
public class UserTransferHelper extends BaseTransferHelper {
    private static final Logger LOG = LoggerFactory.getLogger(UserTransferHelper.class);

    private static UserTransferHelper transferHelper = null;

    private UserTransferHelper(String authServer) {
        super(authServer);
    }

    /**
     * 创建工具实例
     *
     * @param authServer 认证服务器地址
     * @return 工具实例
     */
    public static UserTransferHelper getInstance(@NotBlank String authServer) {
        if (transferHelper == null) {
            synchronized (UserTransferHelper.class) {
                if (transferHelper == null) {
                    transferHelper = new UserTransferHelper(authServer);
                }
            }
        }
        return transferHelper;
    }

    /**
     * 同步用户
     *
     * @param userInfo 用户信息
     * @return 用户ID
     */
    public Result<Long> upsertUser(@NotNull AuthUserDTO userInfo) {
        Assert.notNull(userInfo, "账号信息为空");
        LOG.info("向统一身份认证中心同步账号：{}", userInfo.getUsername());

        // 操作人信息
        if (userInfo.getUpdaterInfo() == null) {
            userInfo.setUpdaterInfo(super.createUpdaterInfo());
        }

        return remoteExchange(CasUrlConstant.URI_USER_UPSERT, HttpMethod.POST, new HttpEntity<>(userInfo),
                new TypeReference<>() {
                });
    }

    /**
     * 批量同步用户
     *
     * @param rollBackAllOnException 出现异常，是否回滚所有
     * @param userInfoList           用户信息列表
     * @return username map userId
     */
    public Result<HashMap<String, Long>> upsertUserBatch(boolean rollBackAllOnException, @NotEmpty List<AuthUserDTO> userInfoList) {
        Assert.notEmpty(userInfoList, "账号信息为空");
        LOG.info("向统一身份认证中心同步账号：{}", userInfoList.stream().map(AuthUserDTO::getUsername).collect(Collectors.toList()));

        // 操作人信息
        for (AuthUserDTO authUserDTO : userInfoList) {
            if (authUserDTO.getUpdaterInfo() == null) {
                authUserDTO.setUpdaterInfo(super.createUpdaterInfo());
            }
        }

        return remoteExchange(CasUrlConstant.URI_USER_UPSERT_BATCH, HttpMethod.POST, new HttpEntity<>(userInfoList),
                new TypeReference<>() {
                }, rollBackAllOnException);
    }

    /**
     * 更新用户状态
     *
     * @param userId  用户ID
     * @param enabled 启用状态
     * @return 用户ID
     */
    public Result<Long> updateEnabled(long userId, boolean enabled) {
        LOG.info("向统一身份认证中心同步账号状态：{}, {}", userId, enabled);

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_ENABLED, HttpMethod.PATCH, null,
                new TypeReference<>() {
                }, userId, enabled);
    }

    /**
     * 修改用户的密码
     *
     * @param userId 用户账号ID
     * @param pd     密码，明文
     * @return 用户ID
     * @deprecated 请使用 {@link #updatePwd(long, String)}
     */
    @Deprecated(forRemoval = true, since = "3.2.5")
    public Result<Long> updatePassword(long userId, @NotBlank String pd) {
        Assert.hasText(pd, "密码为空");
        LOG.info("向统一身份认证中心同步账号密码：{}, {}", userId, pd);

        AuthPwdUpdateDTO updateDTO = new AuthPwdUpdateDTO();
        updateDTO.setUserId(userId);
        updateDTO.setPassword(pd);
        updateDTO.setUpdaterInfo(super.createUpdaterInfo());
        updateDTO.setUpdateType(this.detectePwdRecordTypeEnum());

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_PWD_V1, HttpMethod.PUT, new HttpEntity<>(updateDTO),
                new TypeReference<>() {
                });
    }

    /**
     * 修改用户的密码
     *
     * @param updateDTO 用户账号密码信息
     * @return 用户ID
     */
    public Result<Long> updatePwd(AuthPwdUpdateDTO updateDTO) {
        Assert.notNull(updateDTO, "密码信息为空");
        Assert.notNull(updateDTO.getUserId(), "密码为空");
        Assert.hasText(updateDTO.getPassword(), "密码为空");
        LOG.info("向统一身份认证中心同步账号密码：{}, {}", updateDTO.getUserId(), updateDTO.getPassword());

        if (updateDTO.getUpdaterInfo() == null) {
            updateDTO.setUpdaterInfo(super.createUpdaterInfo());
        }
        if (updateDTO.getUpdateType() == null) {
            updateDTO.setUpdateType(this.detectePwdRecordTypeEnum());
        }

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_PWD_V1, HttpMethod.PUT, new HttpEntity<>(updateDTO),
                new TypeReference<>() {
                });
    }

    /**
     * 校验用户的密码
     *
     * @param userId 用户账号ID
     * @param pwd    当前密码
     * @param pwdNew 新密码
     * @return 用户ID
     */
    public Result<Boolean> validatePwd(Long userId, String pwd, String pwdNew) {
        LOG.info("向统一身份认证中心校验密码：{}, {}，{}", userId, pwd, pwdNew);

        return remoteExchange(CasUrlConstant.URI_USER_VERIFY_PASSWORD, HttpMethod.GET, null,
                new TypeReference<>() {
                }, userId, pwd, pwdNew);
    }

    /**
     * 校验用户的密码
     *
     * @param userId 用户账号ID
     * @param pwd    待校验密码
     * @return 用户ID
     */
    public Result<Boolean> validatePwd(Long userId, String pwd) {
        LOG.info("向统一身份认证中心校验密码：{}, {}", userId, pwd);

        return remoteExchange(CasUrlConstant.URI_USER_VERIFY_PASSWORD_V1, HttpMethod.GET, null,
                new TypeReference<>() {
                }, userId, pwd);
    }

    /**
     * 修改用户的手机号
     *
     * @param userId 用户账号ID
     * @param mobile 手机号
     * @return 用户ID
     */
    public Result<Long> updateMobile(long userId, String mobile) {
        LOG.info("向统一身份认证中心同步手机号：{}, {}", userId, mobile);

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_MOBILE, HttpMethod.PATCH, null,
                new TypeReference<>() {
                }, userId, mobile);
    }

    /**
     * 修改用户的邮箱
     *
     * @param userId 用户账号ID
     * @param email  邮箱
     * @return 用户ID
     */
    public Result<Long> updateEmail(long userId, String email) {
        LOG.info("向统一身份认证中心同步邮箱：{}, {}", userId, email);

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_EMAIL, HttpMethod.PATCH, null,
                new TypeReference<>() {
                }, userId, email);
    }

    /**
     * 修改用户的账号到期时间
     *
     * @param userId      用户账号ID
     * @param expiredTime 账号过期时间，格式yyyy-MM-dd HH:mm:ss
     * @return 用户ID
     */
    public Result<Long> updateExpiredTime(long userId, @NotBlank String expiredTime) {
        Assert.hasText(expiredTime, "过期时间为空");
        LOG.info("向统一身份认证中心同步账号过期时间：{}, {}", userId, expiredTime);

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_EXPIRED_TIME, HttpMethod.PATCH, null,
                new TypeReference<>() {
                }, userId, expiredTime);
    }

    /**
     * 修改用户的密码到期时间
     *
     * @param userId      用户账号ID
     * @param expiredTime 密码过期时间，格式yyyy-MM-dd HH:mm:ss
     * @return 用户ID
     */
    public Result<Long> updatePwdExpiredTime(long userId, String expiredTime) {
        Assert.hasText(expiredTime, "过期时间为空");
        LOG.info("向统一身份认证中心同步密码过期时间：{}, {}", userId, expiredTime);

        return remoteExchange(CasUrlConstant.URI_USER_UPDATE_PWD_EXPIRED_TIME, HttpMethod.PATCH, null,
                new TypeReference<>() {
                }, userId, expiredTime);
    }

    /**
     * 删除用户账号
     *
     * @param userId 用户账号ID
     * @return 用户账号ID
     */
    public Result<Long> delete(long userId) {
        LOG.info("向统一身份认证中心删除账号：{}", userId);

        return remoteExchange(CasUrlConstant.URI_USER_DELETE, HttpMethod.DELETE, null,
                new TypeReference<>() {
                }, userId);
    }

    /**
     * 根据ID获取用户信息
     *
     * @param userId 用户账号ID
     * @return 用户信息
     */
    public Result<AuthUserDTO> getUser(long userId) {
        return remoteExchange(CasUrlConstant.URI_USER_GET, HttpMethod.GET, null,
                new TypeReference<>() {
                }, userId);
    }

    /**
     * 根据username获取用户信息
     *
     * @param username 用户账号
     * @return 用户信息
     */
    public Result<AuthUserDTO> getUserByUsername(@NotBlank String username) {
        return remoteExchange(CasUrlConstant.URI_USER_GET_BY_USERNAME, HttpMethod.GET, null,
                new TypeReference<>() {
                }, username);
    }

    /**
     * 根据手机号获取用户信息
     *
     * @param mobile 手机号
     * @return 用户信息
     */
    public Result<AuthUserDTO> getUserByMobile(@NotBlank String mobile) {
        return remoteExchange(CasUrlConstant.URI_USER_GET_BY_MOBILE, HttpMethod.GET, null,
                new TypeReference<>() {
                }, mobile);
    }

    /**
     * 根据邮箱获取用户信息
     *
     * @param email 邮箱
     * @return 用户信息
     */
    public Result<AuthUserDTO> getUserByEmail(@NotBlank String email) {
        return remoteExchange(CasUrlConstant.URI_USER_GET_BY_EMAIL, HttpMethod.GET, null,
                new TypeReference<>() {
                }, email);
    }

    /**
     * 获取用户列表
     *
     * @param userIds 用户账号ID
     * @return 用户信息
     */
    public Result<ArrayList<AuthUserDTO>> getUserList(@NotEmpty List<Long> userIds) {
        Assert.notEmpty(userIds, "用户信息为空");

        return remoteExchange(CasUrlConstant.URI_USER_LIST, HttpMethod.POST, new HttpEntity<>(userIds),
                new TypeReference<>() {
                });
    }

    /**
     * 获取用户列表
     *
     * @param queryDTO 查询参数
     * @return 用户信息
     */
    public Result<ArrayList<AuthUserDTO>> queryUserList(@NotNull AuthUserQueryDTO queryDTO) {
        Assert.notNull(queryDTO, "查询参数为空");

        return remoteExchange(CasUrlConstant.URI_USER_QUERY, HttpMethod.POST, new HttpEntity<>(queryDTO),
                new TypeReference<>() {
                });
    }

    /**
     * 获取用户ID
     *
     * @param usernames 用户账号
     * @return 用户ID
     */
    public Result<HashMap<String, Long>> getUserIdByUsername(@NotEmpty List<String> usernames) {
        Assert.notEmpty(usernames, "用户账号为空");

        return remoteExchange(CasUrlConstant.URI_USER_ID_BY_USERNAME, HttpMethod.POST, new HttpEntity<>(usernames),
                new TypeReference<>() {
                });
    }

    /**
     * 获取用户ID
     *
     * @param mobiles 手机号
     * @return 用户ID
     */
    public Result<HashMap<String, Long>> getUserIdByMobile(@NotEmpty List<String> mobiles) {
        Assert.notEmpty(mobiles, "用户账号为空");

        return remoteExchange(CasUrlConstant.URI_USER_ID_BY_MOBILE, HttpMethod.POST, new HttpEntity<>(mobiles),
                new TypeReference<>() {
                });
    }

    /**
     * 获取用户ID
     *
     * @param emails 邮箱
     * @return 用户ID
     */
    public Result<HashMap<String, Long>> getUserIdByEmail(@NotEmpty List<String> emails) {
        Assert.notEmpty(emails, "邮箱为空");

        return remoteExchange(CasUrlConstant.URI_USER_ID_BY_EMAIL, HttpMethod.POST, new HttpEntity<>(emails),
                new TypeReference<>() {
                });
    }

    /**
     * 校验客户端用户
     *
     * @param clientId 客户端ID
     * @param userId   用户ID
     * @return 用户信息
     */
    public Result<AuthClientUserDTO> validateClientUser(String clientId, Long userId) {
        return remoteExchange(CasUrlConstant.URI_CLIENT_USER_VALIDATE, HttpMethod.GET, null,
                new TypeReference<>() {
                }, clientId == null ? "" : clientId, userId == null ? "" : userId);
    }

    private PwdRecordTypeEnum detectePwdRecordTypeEnum() {
        var authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication == null || !authentication.isAuthenticated() || AnonymousAuthenticationToken.class.isAssignableFrom(authentication.getClass())) {
            return PwdRecordTypeEnum.RETRIEVE;
        }
        return PwdRecordTypeEnum.UPDATE;
    }
}
