package com.elitescloud.boot.auth.cas.task;

import com.elitescloud.boot.auth.cas.UserSourceProvider;
import com.elitescloud.boot.auth.cas.model.AuthUserDTO;
import com.elitescloud.boot.auth.cas.provider.UserTransferHelper;
import com.elitescloud.boot.auth.util.AuthCompatibilityUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 自动同步用户任务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/15
 */
public class AutoSyncUserTask implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(AutoSyncUserTask.class);
    private static final int MAX_SIZE = 100;

    private final UserSourceProvider userSourceProvider;
    private final UserTransferHelper transferHelper;
    /**
     * 每次查询数据量
     */
    private int syncSize = 50;
    /**
     * 是否一批失败时全部回滚
     */
    private boolean rollbackAllOnException;

    public AutoSyncUserTask(String authServer, UserSourceProvider userSourceProvider) {
        this.userSourceProvider = userSourceProvider;
        this.transferHelper = UserTransferHelper.getInstance(authServer);
    }

    @Override
    public void run() {
        if (userSourceProvider == null) {
            return;
        }

        if (syncSize < 1) {
            // 小于1，则不再同步
            return;
        }
        int size = Math.min(syncSize, MAX_SIZE);

        // 重试次数
        int times = 0;
        while (times < 3) {
            if (!ClientTokenHolder.hasToken()) {
                LOG.info("OAuth2客户端未初始化成功：{}", times + 1);

                try {
                    TimeUnit.MINUTES.sleep(1);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
                times++;
                continue;
            }

            this.startSync(size);
            break;
        }
    }

    private void startSync(int size) {
        // 查询需要同步的用户
        var users = this.queryUserList(size);
        while (!CollectionUtils.isEmpty(users)) {
            // 调用服务端接口同步用户
            var result = transferHelper.upsertUserBatch(rollbackAllOnException, users);
            if (!result.getSuccess()) {
                // 失败则终止
                LOG.error("CAS同步用户账号失败：{}", result.getMsg());
                break;
            }

            // 成功后回调，更新本地记录
            if (!CollectionUtils.isEmpty(result.getData())) {
                userSourceProvider.successCallback(result.getData());
            }

            users = this.queryUserList(size);
        }
    }

    public void setSyncSize(int syncSize) {
        Assert.isTrue(syncSize < MAX_SIZE, "最大值为" + MAX_SIZE);
        this.syncSize = syncSize;
    }

    public void setRollbackAllOnException(boolean rollbackAllOnException) {
        this.rollbackAllOnException = rollbackAllOnException;
    }

    @SuppressWarnings({"deprecation", "removal"})
    private List<AuthUserDTO> queryUserList(int size) {
        // 兼容老的
        if (userSourceProvider instanceof com.elitescloud.cloudt.authorization.sdk.cas.UserSourceProvider) {
            var users = ((com.elitescloud.cloudt.authorization.sdk.cas.UserSourceProvider) userSourceProvider).queryUser(size);
            return AuthCompatibilityUtil.convert(users);
        }

        return userSourceProvider.queryUserList(size);
    }
}
