package com.elitescloud.boot.dubbo.filter;

import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.cloudt.authorization.core.SecurityContextUtil;
import com.elitescloud.cloudt.common.constant.DubboConstant;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.common.extension.Activate;
import org.apache.dubbo.rpc.*;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.server.resource.authentication.JwtAuthenticationToken;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.Collections;

import static org.apache.dubbo.common.constants.CommonConstants.CONSUMER;
import static org.apache.dubbo.common.constants.CommonConstants.PROVIDER;

/**
 * <p>
 * base on Dubbo filter and RpcContext realizationed audit
 * <p>
 *
 * @author Roman.Zhang
 * @date 2021/7/26
 */
@Slf4j
@Activate(group = {CONSUMER, PROVIDER}, order = Integer.MAX_VALUE)
public class DubboAuthenticationContextFilter implements Filter {

    private JwtDecoder jwtDecoder;

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) {
        RpcContext rpcContext = RpcContext.getServiceContext();
        if (rpcContext.getUrl() == null) {
            return invoker.invoke(invocation);
        }

        try {
            if (rpcContext.isConsumerSide()) {
                // 消费者时，传递token
                transportAuthentication(rpcContext);
            } else {
                // 提供者时，获取token
                obtainAuthentication(rpcContext);
            }
            return invoker.invoke(invocation);
        } catch (Exception e) {
            log.error("DubboAuditFilter 异常:" + e.getMessage());
            throw e;
        } finally {
            if (rpcContext.isProviderSide()) {
                // 清掉用户信息
                SecurityContextHolder.clearContext();
            }
        }
    }

    private void transportAuthentication(RpcContext rpcContext) {
        GeneralUserDetails user = SecurityContextUtil.currentUser();
        if (user == null || user.getUser() == null) {
            log.warn("调用dubbo时传递用户信息失败，未获取到当前用户信息");
            return;
        }
        if (user.getUser() != null) {
            // 只有当前信息不为空才需要设置
            rpcContext.setObjectAttachment(DubboConstant.CURRENT_USER_ID, user.getUser().getId().toString());
            rpcContext.setObjectAttachment(DubboConstant.CURRENT_USER_NAME, user.getUser().getUsername());
            if (user.getTenant() != null) {
                rpcContext.setObjectAttachment(DubboConstant.CURRENT_TENANT_ID, user.getTenant().getId().toString());
            }

            String token = SecurityContextUtil.currentToken();
            if (StringUtils.hasText(token)) {
                rpcContext.setAttachment(DubboConstant.CURRENT_AUTH_TOKEN, token);
            }
        }
    }

    private void obtainAuthentication(RpcContext rpcContext) {
        String token = rpcContext.getAttachment(DubboConstant.CURRENT_AUTH_TOKEN);
        if (!StringUtils.hasText(token)) {
            return;
        }

        if (jwtDecoder == null) {
            jwtDecoder = SpringContextHolder.getBean(JwtDecoder.class);
            Assert.notNull(jwtDecoder, "解析token失败，jwtDecoder不存在");
        }
        Jwt jwt = jwtDecoder.decode(token);
        JwtAuthenticationToken authenticationToken = new JwtAuthenticationToken(jwt, Collections.emptyList());
        SecurityContextHolder.getContext().setAuthentication(authenticationToken);
    }
}
