package com.elitescloud.boot.excel.common.spi.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.param.AbstractOrderQueryParam;
import com.elitescloud.boot.excel.common.DataExport;
import com.elitescloud.boot.excel.config.tmpl.DataExportServiceFactory;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.swagger.openapi.common.spi.OpenApiLoadStrategySPI;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.web.bind.annotation.RequestMethod;

import java.lang.reflect.Parameter;
import java.util.*;

/**
 * 加载数据导出为OpenAPI.
 *
 * @author Kaiser（wang shao）
 * @date 2024/4/8
 */
public class OpenApiLoadDataExport implements OpenApiLoadStrategySPI {

    private boolean initialized;
    private final List<Class<?>> exportServiceClasses = new ArrayList<>(32);
    private final Map<Class<?>, DataExportServiceFactory.ServiceMetaData> exportServiceMap = new HashMap<>();
    private final Map<Class<?>, OpenApiInfo> openApiInfoMap = new HashMap<>();

    public OpenApiLoadDataExport() {
        this.initialized = false;
    }

    @Override
    public List<Class<?>> load() {
        if (!initialized) {
            this.init();
        }
        return exportServiceClasses;
    }

    @Override
    public List<OpenApiInfo> analyze(Class<?> clazz) {
        var openApiInfo = openApiInfoMap.get(clazz);
        if (openApiInfo != null) {
            return List.of(openApiInfo);
        }

        if (!DataExport.class.isAssignableFrom(clazz)) {
            return Collections.emptyList();
        }

        var exportService = exportServiceMap.get(clazz);
        if (exportService == null) {
            return Collections.emptyList();
        }
        var method = exportService.getMethod();
        openApiInfo = new OpenApiInfo(method, RequestMethod.POST, "/cloudt/core/import/" + exportService.getDataExport().getTmplCode() + "/export");
        var businessObject = exportService.getDataExportClass().getAnnotation(BusinessObject.class);
        openApiInfo.setBusinessObject(businessObject);

        var businessOperation = method.getAnnotation(BusinessObjectOperation.class);
        if (businessOperation != null) {
            openApiInfo.setBusinessObjectType(businessOperation.businessObjectType());

            if (businessObject == null &&
                    (BusinessObjectOperation.BUSINESS_OBJECT_NAME_DEFAULT.equals(businessOperation.businessObjectType()) ||
                            CharSequenceUtil.isBlank(businessOperation.businessObjectType()))) {
                throw new BusinessException(exportService.getDataExport().getTmplCode() + "对应的导出服务中未指定所属业务对象");
            }
            openApiInfo.setBusinessObjectOperation(businessOperation);
            openApiInfo.setBusinessOperationCode(businessOperation.operationCode());
            openApiInfo.setOperationDescription(businessOperation.operationDescription());
            openApiInfo.setDetail(businessOperation.detail());
            openApiInfo.setOperationType(businessOperation.operationType());
            openApiInfo.setDataPermissionEnabled(businessOperation.dataPermissionEnabled());
            openApiInfo.setFieldPermissionAutoFilter(businessOperation.fieldPermissionAutoFilter());
            openApiInfo.setPermissionRef(businessOperation.permissionRef());
        }

        openApiInfoMap.put(clazz, openApiInfo);
        return List.of(openApiInfo);
    }

    @Override
    public boolean isRequestBody(Parameter parameter) {
        return parameter != null && AbstractOrderQueryParam.class.isAssignableFrom(parameter.getType());
    }

    private void init() {
        synchronized (this) {
            if (initialized) {
                return;
            }

            if (SpringContextHolder.initialized()) {
                // spring容器已初始化
                this.initializeBySpring();
                initialized = true;
            }
        }
    }

    private void initializeBySpring() {
        var factory = SpringContextHolder.getObjectProvider(DataExportServiceFactory.class).getIfAvailable();
        if (factory == null) {
            return;
        }
        var exportService = factory.getAllExportService();
        if (exportService.isEmpty()) {
            return;
        }

        Class<?> clazz = null;
        for (DataExportServiceFactory.ServiceMetaData value : exportService.values()) {
            clazz = AopProxyUtils.ultimateTargetClass(value.getDataExport());
            exportServiceMap.put(clazz, value);
            exportServiceClasses.add(clazz);
        }
    }
}
