package com.elitescloud.boot.excel.config.tmpl.export.strategy;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.common.param.AbstractOrderQueryParam;
import com.elitescloud.boot.excel.config.tmpl.export.ExportStrategyParam;
import com.elitescloud.boot.excel.config.tmpl.export.SystemTmplDataSupport;
import com.elitescloud.cloudt.system.dto.SysImportRateDTO;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StopWatch;

import java.io.Serializable;
import java.util.List;

/**
 * 超出sheet限制后，开启新的sheet.
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/28
 */
class NewSheetExportStrategy <R extends Serializable, P extends AbstractOrderQueryParam> extends BaseExportStrategy<R, P> {
    private static final Logger LOG = LoggerFactory.getLogger(NewSheetExportStrategy.class);

    public NewSheetExportStrategy(FileService<?> fileService, SystemTmplDataSupport fsmTmplSupport, ObjectMapper objectMapper) {
        super(fileService, fsmTmplSupport, objectMapper);
    }

    @Override
    public long export(ExportStrategyParam<R, P> param) {
        int sheetLimit = sheetLimit(param.getTmplDTO());
        int page = 2;
        var pageSize = param.getPageSize();
        long count = 0;
        int sheetIndex = 0;
        int rowCount = 0;

        var excelFile = createExportFile(param);
        var excelWriter = super.createExcelWriter(excelFile);
        WriteSheet sheet = EasyExcel.writerSheet(sheetIndex).sheetName("Sheet " + (sheetIndex + 1)).build();
        // 头部
        var headers = obtainHeaders(param.getTmplDTO());
        excelWriter.write(headers, sheet);

        // 开始写数据
        var pageData = param.getFirstPageData();
        var attributes = obtainAttributes(param.getTmplDTO());
        param.getCostTime().getPrepare().stop();

        while (pageData.isNotEmpty()) {
            if (rowCount == sheetLimit) {
                // 上一页已满
                sheetIndex++;
                rowCount = 0;
                sheet = EasyExcel.writerSheet(sheetIndex).sheetName("Sheet " + (sheetIndex + 1)).build();
                excelWriter.write(headers, sheet);
            }

            var tempDataList = pageData.getRecords();
            if (rowCount + pageData.size() > sheetLimit) {
                // 超出限制了，则将超出部分写入新的sheet
                int start = 0;
                int end = start + (sheetLimit - rowCount);
                tempDataList = pageData.getRecords().subList(start, end);

                // 新的sheet
                while (end < pageData.size()) {
                    // 转换并向excel写入数据
                    super.convertAndWrite(excelWriter, sheet, attributes, tempDataList, param.getCostTime());

                    int newEnd = end + (sheetLimit - rowCount);
                    sheetIndex++;
                    sheet = EasyExcel.writerSheet(sheetIndex).sheetName("Sheet " + (sheetIndex + 1)).build();
                    excelWriter.write(headers, sheet);

                    tempDataList = pageData.getRecords().subList(end, Math.min(pageData.size(), newEnd));
                    rowCount = 0;
                    end = newEnd;
                }
            }
            rowCount += tempDataList.size();
            count += pageData.size();

            // 转换并向excel写入数据
            super.convertAndWrite(excelWriter, sheet, attributes, tempDataList, param.getCostTime());

            // 少于页大小，则说明没有数据了
            var isFinish = pageData.size() < pageSize || pageData.getTotal() == count;
            if (!param.isSync()) {
                // 异步时更新进度
                param.getCostTime().getUpdateResult().start();
                super.updateRate(param.getImportId(), SysImportRateDTO.builder().finish(false).total(pageData.getTotal()).count(count - 1).tmplCode(param.getTmplDTO().getCode()).build());
                param.getCostTime().getUpdateResult().stop();
            }
            if (isFinish) {
                break;
            }

            // 查询下一页数据
            param.getCostTime().getQueryData().start();
            param.getQueryParam().setCurrent(page);
            param.getQueryParam().setSize(pageSize);
            pageData = param.getDataProvider().apply(param.getQueryParam());
            param.getCostTime().getQueryData().stop();
            page++;
        }
        excelWriter.finish();

        // 更新导出结果
        param.getCostTime().getUpdateResult().start();
        updateExportFinish(param, count, excelFile, param.getCostTime());
        excelFile.deleteOnExit();
        param.getCostTime().getUpdateResult().stop();

        return count;
    }
}
