package com.elitescloud.boot.excel.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ReflectUtil;
import com.elitescloud.boot.util.ArrayUtil;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Excel解析的pojo对象.
 *
 * @author Kaiser（wang shao）
 * @date 2024/1/17
 */
public class ExcelBean {
    private static final Map<Class<?>, ExcelBean> CACHE = new HashMap<>(256);

    private final Class<?> clazz;

    private Map<String, BeanField> dataFieldMap;

    private ExcelBean(Class<?> clazz) {
        this.clazz = clazz;
        this.analyze();
    }

    /**
     * 获取实例
     *
     * @param clazz 要解析的对象类型
     * @return 实例对象
     */
    public static ExcelBean getInstance(@NotNull Class<?> clazz) {
        return CACHE.computeIfAbsent(clazz, ExcelBean::new);
    }

    /**
     * 获取字段信息
     *
     * @param fieldName 字段名称
     * @return 字段信息
     */
    public BeanField getField(@NotBlank String fieldName) {
        return dataFieldMap.get(fieldName);
    }

    /**
     * 获取字段上的注解
     *
     * @param fieldName      字段名
     * @param annotationType 注解类型
     * @return 注解列表
     */
    public List<Annotation> getAnnotations(@NotBlank String fieldName, Class<? extends Annotation> annotationType) {
        var annoMap = dataFieldMap.get(fieldName).getFieldAnnotationsMap();
        if (CollUtil.isEmpty(annoMap)) {
            return Collections.emptyList();
        }
        if (annotationType == null) {
            return annoMap.values().stream().flatMap(Collection::stream).collect(Collectors.toList());
        }

        return annoMap.get(annotationType);
    }

    /**
     * 获取字段上的注解
     *
     * @param fieldName      字段名
     * @param annotationType 注解类型
     * @return 注解列表
     */
    @SuppressWarnings("unchecked")
    public <T extends Annotation> T getAnnotation(@NotBlank String fieldName, Class<T> annotationType) {
        var annotations = this.getAnnotations(fieldName, annotationType);
        return CollUtil.isEmpty(annotations) ? null : (T) annotations.get(0);
    }

    private void analyze() {
        // 获取所有的字段
        var fields = ReflectUtil.getFieldMap(clazz);
        if (CollUtil.isEmpty(fields)) {
            this.dataFieldMap = Collections.emptyMap();
            return;
        }

        this.dataFieldMap = new HashMap<>(fields.size());
        for (Map.Entry<String, Field> entry : fields.entrySet()) {
            dataFieldMap.put(entry.getKey(), this.buildBeanField(entry.getValue()));
        }
    }

    private BeanField buildBeanField(Field field) {
        BeanField beanField = new BeanField();
        beanField.setName(field.getName());
        beanField.setField(field);

        var annotationsAll = field.getAnnotations();
        Map<Class<? extends Annotation>, List<Annotation>> fieldAnnotationsMap = null;
        if (ArrayUtil.isEmpty(annotationsAll)) {
            fieldAnnotationsMap = Collections.emptyMap();
        } else {
            fieldAnnotationsMap = new HashMap<>(annotationsAll.length);
            for (Annotation annotation : annotationsAll) {
                fieldAnnotationsMap
                        .computeIfAbsent(annotation.annotationType(), a -> new ArrayList<>(4))
                        .add(annotation);
            }
        }
        beanField.setFieldAnnotationsMap(Collections.unmodifiableMap(fieldAnnotationsMap));

        return beanField;
    }

    public static class BeanField {
        /**
         * 字段名称
         */
        private String name;
        /**
         * 字段
         */
        private Field field;
        /**
         * 字段上的注解
         */
        private Map<Class<? extends Annotation>, List<Annotation>> fieldAnnotationsMap;

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public Field getField() {
            return field;
        }

        public void setField(Field field) {
            this.field = field;
        }

        public Map<Class<? extends Annotation>, List<Annotation>> getFieldAnnotationsMap() {
            return fieldAnnotationsMap;
        }

        public void setFieldAnnotationsMap(Map<Class<? extends Annotation>, List<Annotation>> fieldAnnotationsMap) {
            this.fieldAnnotationsMap = fieldAnnotationsMap;
        }
    }
}
