package com.elitescloud.boot.jpa.config;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ReflectUtil;
import com.elitescloud.cloudt.common.annotation.Comment;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import org.hibernate.boot.spi.InFlightMetadataCollector;
import org.hibernate.boot.spi.MetadataContributor;
import org.hibernate.mapping.Column;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Table;
import org.jboss.jandex.IndexView;
import org.springframework.util.StringUtils;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * hibernate元数据处理.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/26
 */
public class CloudtMetadataContributor implements MetadataContributor {

    @Override
    public void contribute(InFlightMetadataCollector metadataCollector, IndexView jandexIndex) {
        for (PersistentClass entityBinding : metadataCollector.getEntityBindings()) {
            processComment(entityBinding);
        }
    }

    private void processComment(PersistentClass persistentClass) {
        var table = persistentClass.getTable();

        // 表注释
        var entityClass = persistentClass.getMappedClass();
        if (!StringUtils.hasText(table.getComment())) {
            table.setComment(obtainTableComment(persistentClass, entityClass));
        }

        // 列注释
        fillColumn(table, entityClass);
    }

    private void fillColumn(Table table, Class<?> entityClass) {
        Iterator<Column> columnIterable = table.getColumnIterator();
        Map<String, Column> columnMap = new HashMap<>(64);
        Column column = null;
        while (columnIterable.hasNext()) {
            column = columnIterable.next();
            columnMap.put(column.getName().toLowerCase().replace("_", ""), column);
        }
        if (columnMap.isEmpty()) {
            return;
        }

        var fields = ReflectUtil.getFields(entityClass);
        for (Field field : fields) {
            column = columnMap.get(field.getName().toLowerCase().replace("_", ""));
            if (column == null) {
                continue;
            }

            // 字段注释
            String fieldComment = obtainFieldComment(field);
            column.setComment(fieldComment);

            // 默认值
            column.setDefaultValue(obtainDefaultValue(field));
        }
    }

    private String obtainDefaultValue(Field field) {
        Comment comment = field.getAnnotation(Comment.class);
        if (comment != null) {
            return normalizeValue(field, comment.defaultValue());
        }

        return null;
    }

    private String normalizeValue(Field field, String defValue) {
        if (CharSequenceUtil.isBlank(defValue)) {
            return null;
        }

        var fieldType = field.getType();
        if (fieldType == String.class || fieldType.isEnum()) {
            return defValue.startsWith("'") ? defValue : "'" + defValue + "'";
        }
        return defValue;
    }

    private String obtainFieldComment(Field field) {
        // 自定义注释
        Comment comment = field.getAnnotation(Comment.class);
        if (comment != null && StringUtils.hasText(comment.value())) {
            return comment.value();
        }

        // hibernate注释
        org.hibernate.annotations.Comment commentHibernate = field.getAnnotation(org.hibernate.annotations.Comment.class);
        if (commentHibernate != null && StringUtils.hasText(commentHibernate.value())) {
            return commentHibernate.value();
        }

        // swagger注释
        ApiModelProperty apiModelProperty = field.getAnnotation(ApiModelProperty.class);
        if (apiModelProperty != null && StringUtils.hasText(apiModelProperty.value())) {
            return apiModelProperty.value();
        }

        return null;
    }

    private String obtainTableComment(PersistentClass persistentClass, Class<?> entityClass) {
        // 自定义注释
        Comment comment = entityClass.getAnnotation(Comment.class);
        if (comment != null && StringUtils.hasText(comment.value())) {
            return comment.value();
        }

        // swagger注释
        ApiModel apiModel = entityClass.getAnnotation(ApiModel.class);
        if (apiModel != null) {
            String value = apiModel.value();
            if (!StringUtils.hasText(value)) {
                value = apiModel.description();
            }
            if (StringUtils.hasText(value)) {
                return value;
            }
        }
        return null;
    }
}
