package com.elitescloud.boot.jpa.support.id.provider.snowflake.assigner;

import com.elitescloud.boot.jpa.support.id.config.IdProperties;
import com.elitescloud.boot.jpa.support.id.provider.snowflake.WorkerInfo;
import com.elitescloud.boot.jpa.support.id.provider.snowflake.config.SnowflakeProperties;
import com.elitescloud.boot.jpa.support.id.provider.snowflake.micro.Snowflake;
import lombok.extern.log4j.Log4j2;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.util.StringUtils;

import java.time.Duration;

/**
 * 基于Redis.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/7
 */
@Log4j2
public class RedisSnowflakeWorkerAssigner extends AbstractSnowflakeWorkerAssigner {
    private static String CACHE_KEY = "cloudt_snowflake";

    private final String dataCenterName;
    private final IdProperties idProperties;
    private final RedisTemplate redisTemplate;

    private Long dataCenterId;
    private Long workerId;

    public RedisSnowflakeWorkerAssigner(String dataCenterName, IdProperties idProperties, RedisTemplate redisTemplate) {
        this.dataCenterName = dataCenterName;
        this.idProperties = idProperties;
        this.redisTemplate = redisTemplate;
    }

    @Override
    protected WorkerInfo distribute() {
        SnowflakeProperties snowflakeProperties = idProperties.getSnowflake();
        if (StringUtils.hasText(snowflakeProperties.getCachePrefix())) {
            CACHE_KEY = snowflakeProperties.getCachePrefix();
        }

        // 先生成dataCenterId
        generateDataCenterId();

        // 再生成workerId
        generateWorkerId();

        return new WorkerInfo(dataCenterId, workerId);
    }

    @Override
    public void refreshAlive() {
        var opsValue = redisTemplate.opsForValue();

        Duration alive = idProperties.getAlive();
        opsValue.set(cacheKeyDataCenterID(), dataCenterId.toString(), alive);
        opsValue.set(cacheKeyDataCenterID(dataCenterId), dataCenterName, alive);
        opsValue.set(cacheKeyWorkerId(workerId), getIp(), alive);
    }

    @Override
    public void destroy() {
        try {
            // 删除当前的workerId
            redisTemplate.delete(cacheKeyWorkerId(workerId));
        } catch (Exception e) {
            log.error("销毁IdGenerator异常：", e);
        }
        log.info("ID生成器实例销毁：{}【{}-{}】", dataCenterName, dataCenterId, workerId);
    }

    private void generateDataCenterId() {
        // 先查询是否已存在对应应用的dataCenterId
        dataCenterId = existsDataCenterId();
        if (dataCenterId != null) {
            // 已存在，则直接复用
            return;
        }

        // 不存在则生成
        long max = Snowflake.getMaxDataCenterId();
        long temp = 0;
        var opsValue = redisTemplate.opsForValue();
        while (temp < max) {
            // 尝试使用该key
            Boolean result = opsValue.setIfAbsent(cacheKeyDataCenterID(temp), dataCenterName,
                    idProperties.getAlive());
            if (Boolean.TRUE.equals(result)) {
                // temp可用，则再次检查是否已占用
                checkDataCenterId(temp);
                break;
            }

            temp++;
        }
        if (temp >= max) {
            throw new IllegalStateException("dataCenterId超过限制" + max);
        }
    }

    private void checkDataCenterId(Long tempDataCenterId) {
        // 如果已有其它实例设置，则取其它实例设置的
        Boolean result = redisTemplate.opsForValue().setIfAbsent(cacheKeyDataCenterID(), tempDataCenterId.toString(),
                idProperties.getAlive());
        if (Boolean.TRUE.equals(result)) {
            dataCenterId = tempDataCenterId;
            return;
        }
        dataCenterId = existsDataCenterId();

        // 其它应用已生成，则删除当前生成的
        redisTemplate.delete(cacheKeyDataCenterID(tempDataCenterId));
    }

    private Long existsDataCenterId() {
        Object val = redisTemplate.opsForValue().get(cacheKeyDataCenterID());
        if (val == null || val instanceof Long) {
            return (Long) val;
        }

        return Long.parseLong(val.toString());
    }

    private void generateWorkerId() {
        long max = Snowflake.getMaxWorkerId();
        long temp = 0;
        var opsValue = redisTemplate.opsForValue();
        var ip = super.getIp();
        while (temp < max) {
            Boolean result = opsValue.setIfAbsent(cacheKeyWorkerId(temp), ip, idProperties.getAlive());
            if (Boolean.TRUE.equals(result)) {
                // workerId可用
                workerId = temp;
                return;
            }

            temp++;
        }
        throw new IllegalStateException("workerId超过限制" + max);
    }

    private String cacheKeyDataCenterID() {
        return CACHE_KEY + ":" + dataCenterName;
    }

    private String cacheKeyDataCenterID(Long theDataCenterId) {
        return CACHE_KEY + ":" + "dataCenter" + "_" + theDataCenterId;
    }

    private String cacheKeyWorkerId(Long theWorkerId) {
        return CACHE_KEY + ":" + "worker" + "_" + dataCenterId + "_" + theWorkerId;
    }

}
