package com.elitescloud.boot.openfeign.config;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.ProxyUtil;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import feign.Feign;
import feign.Target;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.openfeign.FeignClientFactoryBean;
import org.springframework.cloud.openfeign.FeignContext;
import org.springframework.cloud.openfeign.Targeter;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Map;
import java.util.Set;

/**
 * 执行目标代理.
 *
 * @author Kaiser（wang shao）
 * @date 2023/7/18
 */
class TargeterProxy implements Targeter {

    private final Targeter proxy;
    private final Set<String> localServiceIds;
    private final CloudtOpenFeignProperties properties;

    public TargeterProxy(Targeter proxy, Set<String> localServiceIds, CloudtOpenFeignProperties properties) {
        this.proxy = proxy;
        this.localServiceIds = localServiceIds;
        this.properties = properties;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T target(FeignClientFactoryBean factory, Feign.Builder feign, FeignContext context, Target.HardCodedTarget<T> target) {
        var instance = proxy.target(factory, feign, context, target);
        if (this.needProxy(target.name())) {
            // 创建本地代理
            return (T) Proxy.newProxyInstance(target.getClass().getClassLoader(),
                    new Class[]{target.type()}, new LocalBeanTarget(target.type(), instance, properties));
        }

        // 返回原代理
        return instance;
    }

    private boolean needProxy(String targetName) {
        if (CollUtil.isEmpty(localServiceIds)) {
            return false;
        }
        boolean proxyAll = localServiceIds.stream().anyMatch("*"::equals);
        if (proxyAll) {
            return true;
        }

        if (CharSequenceUtil.isBlank(targetName)) {
            return false;
        }

        return localServiceIds.contains(targetName);
    }

    @Slf4j
    static class LocalBeanTarget implements InvocationHandler {
        /**
         * 客户端接口
         */
        private final Class<?> feignClientInterface;
        /**
         * 原openfeign代理实例
         */
        private final Object feignProxy;
        private final CloudtOpenFeignProperties properties;
        /**
         * 初始化后真正要执行的实例对象
         */
        private Object bean = null;
        /**
         * 是否已初始化
         */
        private boolean initialized = false;

        public LocalBeanTarget(Class<?> clazz, Object feignInstance, CloudtOpenFeignProperties properties) {
            this.feignClientInterface = clazz;
            this.feignProxy = feignInstance;
            this.properties = properties;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            // 内置方法执行
            switch (method.getName()) {
                case "equals":
                    try {
                        Object otherHandler =
                                args.length > 0 && args[0] != null ? Proxy.getInvocationHandler(args[0]) : null;
                        return equals(otherHandler);
                    } catch (IllegalArgumentException e) {
                        return false;
                    }
                case "hashCode":
                    return hashCode();
                case "toString":
                    return "cloudtFeignClient_local_" + feignClientInterface.getName();
                default:
            }

            // 禁用使用本地的
            if (Boolean.FALSE.equals(properties.getDirectLocal())) {
                return method.invoke(feignProxy, args);
            }

            // 已初始化过
            if (initialized) {
                return this.invokeLocal(bean, method, args);
            }

            // spring容器未初始化完，则走原rpc逻辑
            if (!SpringContextHolder.initialized()) {
                return method.invoke(feignProxy, args);
            }

            // 寻找本地bean实例
            for (Map.Entry<String, ?> entry : SpringContextHolder.getBeans(feignClientInterface).entrySet()) {
                if (!(ProxyUtil.getTargetBeanForSpring(entry.getValue()) instanceof Proxy)) {
                    // 内部实现的bean
                    bean = entry.getValue();
                    initialized = true;

                    log.info("{} used local bean：{}", feignClientInterface.getName(), entry.getKey());
                    break;
                }
            }
            if (!initialized) {
                // 默认使用RPC的
                bean = feignProxy;
                initialized = true;
            }

            return this.invokeLocal(bean, method, args);
        }

        private Object invokeLocal(Object bean, Method method, Object[] args) throws InvocationTargetException, IllegalAccessException {
            if (method.getReturnType().isAssignableFrom(ApiResult.class)) {
                try {
                    return method.invoke(bean, args);
                } catch (Throwable e) {
                    Throwable exp = this.getTargetException(e);
                    log.error("调用RPC接口异常：", exp);
                    if (exp instanceof BusinessException) {
                        var businessException = (BusinessException) exp;
                        var errorCode = ObjectUtil.defaultIfNull(businessException.getCode(), ApiCode.FAIL.getCode());
                        return ApiResult.fail(errorCode, exp.getMessage());
                    }
                    return ApiResult.fail(exp.getMessage());
                }
            }
            return method.invoke(bean, args);
        }

        private Throwable getTargetException(Throwable e) {
            Throwable cause = e.getCause();
            while (cause != null) {
                if (cause instanceof ReflectiveOperationException) {
                    cause = cause.getCause();
                    continue;
                }
                break;
            }
            return cause;
        }
    }
}
