package com.elitescloud.boot.openfeign.common;

import org.springframework.cloud.openfeign.FeignClientBuilder;
import org.springframework.context.ApplicationContext;
import org.springframework.lang.NonNull;
import org.springframework.util.StringUtils;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 动态调用客户端.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/1
 */
public class DynamicClientHelper {
    private final FeignClientBuilder feignClientBuilder;

    private static final Map<String, Object> CLIENT_CACHE = new ConcurrentHashMap<>();

    public DynamicClientHelper(ApplicationContext applicationContext) {
        this.feignClientBuilder = new FeignClientBuilder(applicationContext);
    }

    /**
     * 获取服务客户端
     *
     * @param serviceName 服务名称
     * @param clazz       服务接口类型
     * @param path        服务接口路径
     * @param <T>         接口类型
     * @return 服务接口实例
     */
    @SuppressWarnings("unchecked")
    public <T> T getClient(@NonNull String serviceName, @NonNull Class<T> clazz, String path) {
        String key = serviceName + ":" + clazz.getName();
        Object client = CLIENT_CACHE.get(key);
        if (client != null) {
            return (T) client;
        }

        // 如果缓存没有，则构建
        synchronized (CLIENT_CACHE) {
            client = CLIENT_CACHE.get(key);
            if (client != null) {
                return (T) client;
            }

            FeignClientBuilder.Builder<T> clientBuilder = feignClientBuilder.forType(clazz, serviceName);
            if (StringUtils.hasText(path)) {
                clientBuilder.path(path);
            }
            client = clientBuilder.build();

            CLIENT_CACHE.put(key, client);
        }
        return (T) client;
    }
}
