package com.elitescloud.boot.openfeign.config;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.openfeign.CloudtOpenFeignProperties;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.cloud.openfeign.Targeter;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.lang.NonNull;
import org.springframework.util.StringUtils;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * OpenFeign的bean后置处理器.
 * <p>
 * 在OpenFeign的{@link org.springframework.cloud.openfeign.FeignClientsRegistrar#registerFeignClient(BeanDefinitionRegistry, AnnotationMetadata, Map)}处通过FeignClientFactoryBean
 * 来给所有FeignClient生成bean，其类型为Targeter。通过代理替换其targeter对象，判断如果是本地已有实现bean，走直接使用本地的spring bean，而不是使用openfeign的。
 *
 * @author Kaiser（wang shao）
 * @date 2023/7/18
 */
public class OpenFeignBeanPostProcessor implements BeanPostProcessor {

    private final String applicationName;
    private final CloudtOpenFeignProperties properties;

    public OpenFeignBeanPostProcessor(String applicationName, CloudtOpenFeignProperties properties) {
        this.applicationName = applicationName;
        this.properties = properties;
    }

    @Override
    public Object postProcessAfterInitialization(@NonNull Object bean, @NonNull String beanName) throws BeansException {
        if (bean instanceof Targeter) {
            return new TargeterProxy((Targeter) bean, this.buildProxyServices(), properties);
        }
        return BeanPostProcessor.super.postProcessAfterInitialization(bean, beanName);
    }

    private Set<String> buildProxyServices() {
        Set<String> services = new HashSet<>();
        if (CollUtil.isNotEmpty(properties.getDirectLocalServices())) {
            // 是否代理所有
            boolean proxyAll = properties.getDirectLocalServices().stream().anyMatch("*"::equals);
            if (proxyAll) {
                services.add("*");
                return services;
            }

            services.addAll(properties.getDirectLocalServices());
        }

        // 代理当前应用
        if (StringUtils.hasText(applicationName)) {
            services.add(applicationName);
        }

        return services;
    }
}
