package com.elitescloud.boot.spi.registrar;

import com.elitescloud.boot.spi.common.SpiService;
import com.elitescloud.boot.spi.common.BaseSpiService;
import com.elitescloud.boot.spi.support.SpiServiceProxy;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.util.Assert;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Proxy;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * SPI服务实例工厂.
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/11
 */
@Log4j2
public class SpiServiceFactory<T extends BaseSpiService<T>> implements FactoryBean<T> {
    private final Class<T> spiService;
    private final SpiService spiServiceAnnotation;
    private final Class<?> spiClass;
    private final List<Class<?>> configSpiInstances;

    public SpiServiceFactory(Class<T> spiService, SpiService spiServiceAnnotation, Map<Class<?>, List<Class<?>>> spiInstances) {
        this.spiService = spiService;
        this.spiServiceAnnotation = spiServiceAnnotation;
        this.spiClass = obtainTargetSpiClass();
        this.configSpiInstances = spiInstances.getOrDefault(spiClass, Collections.emptyList());
    }

    @SuppressWarnings("unchecked")
    @Override
    public T getObject() throws Exception {
        return (T) Proxy.newProxyInstance(spiService.getClassLoader(), new Class[]{spiService},
                new SpiServiceProxy(spiServiceAnnotation, spiClass, configSpiInstances));
    }

    @Override
    public Class<?> getObjectType() {
        return spiService;
    }

    private Class<?> obtainTargetSpiClass() {
        var types = spiService.getGenericInterfaces();
        Assert.notEmpty(types, "未获取到" + spiService.getName() + "继承的接口");

        for (Type type : types) {
            if (!(type instanceof ParameterizedType)) {
                continue;
            }
            ParameterizedType parameterizedType = (ParameterizedType) type;
            if (parameterizedType.getRawType() == BaseSpiService.class) {
                var typeArguments = ((ParameterizedType) type).getActualTypeArguments();
                if (typeArguments != null && typeArguments.length > 0) {
                    return (Class<?>) typeArguments[0];
                }
                throw new IllegalStateException(spiService.getName() + "的" + BaseSpiService.class.getSimpleName() + "未指定SPI接口类型");
            }
        }
        throw new IllegalStateException(spiService.getName() + "未继承" + BaseSpiService.class.getSimpleName());
    }
}