package com.elitescloud.boot.spi.registrar;

import com.elitescloud.boot.spi.common.SpiService;
import com.elitescloud.boot.spi.common.BaseSpiService;
import com.elitescloud.boot.spi.CloudtSpiProperties;
import com.elitescloud.cloudt.context.util.PropertiesUtil;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.*;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.env.Environment;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.core.type.classreading.MetadataReader;
import org.springframework.core.type.classreading.MetadataReaderFactory;
import org.springframework.core.type.filter.AssignableTypeFilter;
import org.springframework.util.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * SPI服务自动注册.
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/10
 */
@Log4j2
public class SpiServiceRegistrar implements ImportBeanDefinitionRegistrar, BeanFactoryAware, EnvironmentAware {
    private static final String PROPERTIES = "META-INF/CloudtSpi.properties";

    private BeanFactory beanFactory;
    private Environment environment;

    @Override
    public void setBeanFactory(@Nonnull BeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }

    @Override
    public void setEnvironment(@Nonnull Environment environment) {
        this.environment = environment;
    }

    @Override
    public void registerBeanDefinitions(@Nonnull AnnotationMetadata importingClassMetadata, @Nonnull BeanDefinitionRegistry registry,
                                        @Nullable BeanNameGenerator importBeanNameGenerator) {
        var packages = obtainScanPackages();
        if (packages.isEmpty()) {
            log.warn("未发现有效的SPI服务接口的有效包路径");
            return;
        }

        var definitions = scanBeanDefinition(packages);

        // 开始注册bean
        for (BeanDefinition definition : definitions) {
            Class<?> serviceClass = null;
            try {
                serviceClass = Class.forName(definition.getBeanClassName());
            } catch (ClassNotFoundException e) {
                throw new RuntimeException(e);
            }
            var spiService = serviceClass.getAnnotation(SpiService.class);
            String beanName = generateBeanName(definition, spiService);

            // 注册bean
            registry.registerBeanDefinition(beanName, buildBeanDefinition(serviceClass, spiService));
            log.info("注册SpiService：{}", serviceClass.getName());
        }
    }

    @Override
    public void registerBeanDefinitions(@Nonnull AnnotationMetadata importingClassMetadata, @Nonnull BeanDefinitionRegistry registry) {
        this.registerBeanDefinitions(importingClassMetadata, registry, null);
    }

    private BeanDefinition buildBeanDefinition(Class<?> serviceClass, SpiService spiService) {
        GenericBeanDefinition beanDefinition = (GenericBeanDefinition) BeanDefinitionBuilder.genericBeanDefinition(serviceClass).getRawBeanDefinition();
        beanDefinition.setBeanClass(SpiServiceFactory.class);
        beanDefinition.getConstructorArgumentValues().addGenericArgumentValue(serviceClass);
        beanDefinition.getConstructorArgumentValues().addGenericArgumentValue(spiService);

        // 加载配置文件配置的SPI实现实例
        var spiInstances = this.loadSpiProperties();
        beanDefinition.getConstructorArgumentValues().addGenericArgumentValue(spiInstances);

        beanDefinition.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_BY_TYPE);
        return beanDefinition;
    }

    private String generateBeanName(BeanDefinition beanDefinition, SpiService spiService) {
        if (spiService != null && StringUtils.hasText(spiService.beanName())) {
            return spiService.beanName();
        }

        return beanDefinition.getBeanClassName() + "SpiService";
    }

    private Set<BeanDefinition> scanBeanDefinition(Set<String> basePackages) {
        Set<BeanDefinition> consumers = new HashSet<>();
        var componentProvider = new CustomClassPathScanningCandidateComponentProvider();
        componentProvider.addIncludeFilter(new CustomTypeFilter(BaseSpiService.class));

        for (String basePackage : basePackages) {
            var beanDefinitions = componentProvider.findCandidateComponents(basePackage.replace(".", "/"));
            consumers.addAll(beanDefinitions);
        }

        return consumers;
    }

    private Set<String> obtainScanPackages() {
        var packages = environment.getProperty(CloudtSpiProperties.CONFIG_PREFIX + ".scan-spi-service-packages", "");
        if (StringUtils.hasText(packages)) {
            return Arrays.stream(packages.split(",")).filter(StringUtils::hasText).collect(Collectors.toSet());
        }

        // 取启动类所在包
        if (!AutoConfigurationPackages.has(this.beanFactory)) {
            return Collections.emptySet();
        }
        return new HashSet<>(AutoConfigurationPackages.get(this.beanFactory));
    }

    private Map<Class<?>, List<Class<?>>> loadSpiProperties() {
        // 加载配置获取所有的接口及实现名称
        var properties = PropertiesUtil.loadResourceProperties(PROPERTIES, null);
        if (properties.isEmpty()) {
            return Collections.emptyMap();
        }

        // 加载接口的实现
        var classLoader = this.getClass().getClassLoader();
        Map<Class<?>, List<Class<?>>> configInstances = new HashMap<>(128);

        try {
            for (Map.Entry<String, List<String>> entry : properties.entrySet()) {
                Class<?> spi = classLoader.loadClass(entry.getKey());
                List<Class<?>> instances = entry.getValue().stream()
                        .filter(StringUtils::hasText)
                        .map(t -> {
                            try {
                                return classLoader.loadClass(t);
                            } catch (ClassNotFoundException e) {
                                throw new IllegalStateException("加载SPI[" + entry.getKey() + "]实现" + t + "异常：", e);
                            }
                        })
                        .collect(Collectors.toList());

                if (!instances.isEmpty()) {
                    configInstances.put(spi, instances);
                }
            }
        } catch (ClassNotFoundException e) {
            throw new IllegalStateException("加载SPI实现异常：", e);
        }
        return configInstances;
    }

    static class CustomClassPathScanningCandidateComponentProvider extends ClassPathScanningCandidateComponentProvider {
        public CustomClassPathScanningCandidateComponentProvider() {
            super(false);
        }

        @Override
        protected boolean isCandidateComponent(@Nonnull AnnotatedBeanDefinition beanDefinition) {
            return true;
        }
    }

    static class CustomTypeFilter extends AssignableTypeFilter {

        public CustomTypeFilter(Class<?> targetType) {
            super(targetType);
        }

        @Override
        public boolean match(MetadataReader metadataReader, @Nonnull MetadataReaderFactory metadataReaderFactory)
                throws IOException {
            return metadataReader.getClassMetadata().isInterface()
                    && !metadataReader.getClassMetadata().getClassName().equals(getTargetType().getName())
                    && super.match(metadataReader, metadataReaderFactory);
        }
    }
}
