package com.elitescloud.boot.spi.support;

import cn.hutool.core.util.ArrayUtil;
import com.elitescloud.boot.spi.common.BaseSpiService;
import com.elitescloud.boot.spi.common.InstanceStrategy;
import com.elitescloud.boot.spi.common.SpiService;
import com.elitescloud.boot.spi.CloudtSpiProperties;
import com.elitescloud.boot.spi.exception.SpiException;
import com.elitescloud.boot.spi.registrar.SpiServiceFactory;
import com.elitescloud.boot.spi.strategy.DefaultInstanceStrategy;
import com.elitescloud.cloudt.common.annotation.context.spi.SpiInstance;
import com.elitescloud.boot.SpringContextHolder;
import org.springframework.core.annotation.OrderUtils;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;
import java.util.stream.Collectors;

/**
 * SPIService代理.
 *
 * @author Kaiser（wang shao）
 * @date 2022/11/11
 */
public class SpiServiceProxy<T> implements InvocationHandler {

    private final SpiService spiService;
    /**
     * SPI接口
     */
    private final Class<T> spiTarget;

    /**
     * 配置文件CloudtSpi.properties中配置的实例
     */
    private final List<Class<T>> configSpiInstances;

    /**
     * 所有实例
     */
    private List<InstanceStrategy.InstanceWrapper<T>> instanceList;
    /**
     * 默认实例
     */
    private List<T> instanceDefault;

    /**
     * 实例选择策略
     */
    private InstanceStrategy instanceStrategy;

    public SpiServiceProxy(@Nullable SpiService spiService, @Nonnull Class<T> spiTarget, @Nullable List<Class<T>> configSpiInstances) {
        this.spiService = spiService;
        this.spiTarget = spiTarget;
        this.configSpiInstances = configSpiInstances;
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        if (CollectionUtils.isEmpty(instanceList)) {
            // 加载实例
            try {
                loadInstance();
            } catch (Exception e) {
                instanceList.clear();
                throw e;
            }
        }
        if (instanceStrategy == null) {
            // 加载策略
            loadStrategy();
        }

        // 选择实例
        var instanceWrapperList = instanceList.stream().map(t -> new InstanceStrategy.InstanceWrapper<>(t.getInstance(), t.isConfigFile(), t.getOrder(), t.getTags()))
                .collect(Collectors.toList());
        var instances = instanceStrategy.choose(instanceWrapperList);
        if (CollectionUtils.isEmpty(instances)) {
            instances = instanceDefault;
        }
        if (instances.isEmpty()) {
            if (spiService != null && spiService.throwExceptionOnNoInstance()) {
                throw new SpiException("执行失败，" + spiTarget.getName() + "无有效实例");
            }
            return null;
        }

        // 执行
        Object result = null;
        for (Object instance : instances) {
            result = method.invoke(instance, args);
        }
        return result;
    }

    private void loadInstance() {
        synchronized (SpiServiceFactory.class) {
            if (!CollectionUtils.isEmpty(instanceList)) {
                return;
            }
            instanceList = new ArrayList<>(8);
            instanceDefault = new ArrayList<>(8);
            Set<String> existsInstances = new HashSet<>(8);

            var primaryClassNames = obtainPrimaryClassName();
            // spi加载
            for (ProviderWrapper t : ProviderInstanceHolder.findProvider(spiTarget, false)) {
                if (t.getProvider() instanceof BaseSpiService) {
                    continue;
                }
                InstanceStrategy.InstanceWrapper<T> instanceWrapper = toInstanceWrapper((T) t.getProvider(), false, t.getOrder());
                instanceList.add(instanceWrapper);
                existsInstances.add(t.getClazz().getName());

                // 默认实例
                if (t.isPrimary()) {
                    instanceDefault.add(instanceWrapper.getInstance());
                } else if (primaryClassNames.contains(t.getClazz().getName())) {
                    instanceDefault.add(instanceWrapper.getInstance());
                }
            }

            // 配置文件加载
            for (Class<T> clazz : configSpiInstances) {
                var className = clazz.getName();
                if (existsInstances.contains(className)) {
                    // 已存在
                    continue;
                }

                InstanceStrategy.InstanceWrapper<T> instanceWrapper = toInstanceWrapper(buildInstance(clazz), true, OrderUtils.getOrder(clazz, 0));
                instanceList.add(instanceWrapper);
                existsInstances.add(className);

                // 默认实例
                if (primaryClassNames.contains(className)) {
                    instanceDefault.add(instanceWrapper.getInstance());
                }
            }

            Assert.state(!instanceList.isEmpty(), "未找到" + spiTarget.getName() + "存在有效实例");
            // 默认实例
            if (spiService != null) {
                if (spiService.primaryRequired()) {
                    Assert.state(!instanceDefault.isEmpty(), spiTarget.getName() + "未发现默认实例");
                }
                if (spiService.primaryUnique()) {
                    Assert.state(instanceDefault.size() <= 1, spiTarget.getName() + "发现多个默认实例");
                }
            }
        }
    }

    private T buildInstance(Class<T> instanceClass) {
        var bean = SpringContextHolder.getObjectProvider(instanceClass).getIfAvailable();
        if (bean != null) {
            return (T) bean;
        }

        T instance = null;
        try {
            instance = instanceClass.getConstructor().newInstance();
        } catch (Exception e) {
            throw new IllegalStateException(instanceClass.getName() + "缺少一个无参构造方法");
        }
        // 注入属性
        SpringContextHolder.autowireBean(instance);
        return instance;
    }

    private InstanceStrategy.InstanceWrapper<T> toInstanceWrapper(T provider, boolean configFile, int order) {
        Set<String> tags = new HashSet<>();
        SpiInstance spiInstance = SpiUtil.obtainAnnotationSingle(provider.getClass(), SpiInstance.class);
        if (spiInstance != null && ArrayUtil.isNotEmpty(spiInstance.tags())) {
            tags.addAll(Arrays.asList(spiInstance.tags()));
        }
        return new InstanceStrategy.InstanceWrapper<>(provider, configFile, order, Collections.unmodifiableSet(tags));
    }

    private void loadStrategy() throws NoSuchMethodException, InvocationTargetException, InstantiationException, IllegalAccessException, ClassNotFoundException {
        if (spiService == null) {
            // SpiService未配置，则寻找默认策略
            var defaultInstanceStrategy = SpringContextHolder.getBean(CloudtSpiProperties.class).getDefaultInstanceStrategy();
            if (StringUtils.hasText(defaultInstanceStrategy)) {
                this.instanceStrategy = (InstanceStrategy) Class.forName(defaultInstanceStrategy).getDeclaredConstructor().newInstance();
                return;
            }

            // 默认策略
            this.instanceStrategy = new DefaultInstanceStrategy();
            return;
        }
        if (StringUtils.hasText(spiService.instanceStrategyName())) {
            // 加载spring bean
            this.instanceStrategy = SpringContextHolder.getBean(spiService.instanceStrategyName(), InstanceStrategy.class);
            return;
        }

        this.instanceStrategy = spiService.instanceStrategy().getDeclaredConstructor().newInstance();
    }

    private Set<String> obtainPrimaryClassName() {
        if (spiService == null) {
            return Collections.emptySet();
        }
        Set<String> classNames = new HashSet<>(8);
        if (ArrayUtil.isNotEmpty(spiService.primary())) {
            classNames.addAll(Arrays.asList(spiService.primary()));
        }

        if (ArrayUtil.isNotEmpty(spiService.primaryClass())) {
            for (Class<?> clazz : spiService.primaryClass()) {
                classNames.add(clazz.getName());
            }
        }
        return classNames;
    }
}
