package com.elitescloud.boot.tenant.client.support.impl;

import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.wrapper.Isolatable;
import com.elitescloud.boot.wrapper.TransactionWrapper;
import com.elitescloud.cloudt.core.tenant.support.TenantDataIsolateProvider;
import com.elitescloud.cloudt.system.constant.TenantType;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.extern.log4j.Log4j2;

import java.util.List;
import java.util.function.Supplier;

/**
 * 租户数据隔离工具.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/11
 */
@Log4j2
public class DefaultTenantDataIsolateProvider implements TenantDataIsolateProvider {
    private final TransactionWrapper transactionWrapper;
    private final TenantClientProvider tenantClientProvider;

    public DefaultTenantDataIsolateProvider(TransactionWrapper transactionWrapper, TenantClientProvider tenantClientProvider) {
        this.transactionWrapper = transactionWrapper;
        this.tenantClientProvider = tenantClientProvider;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T byNone(Supplier<T> supplier) {
        return (T) transactionWrapper.apply(supplier, Isolatable.NONE);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T byDefaultDirectly(Supplier<T> supplier) {
        return (T) transactionWrapper.apply(supplier, null);
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> T byTenantDirectly(Supplier<T> supplier, SysTenantDTO tenantDTO) {
        return (T) transactionWrapper.apply(supplier, tenantDTO);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T byTenantUser(Supplier<T> supplier, SysUserDTO currentUser) {
        if (currentUser.getSysTenantVO() != null && currentUser.getSysTenantVO().getSysUserId().longValue() != currentUser.getId()
                && currentUser.getSysTenantVO().getType() != TenantType.OPERATION) {
            return (T) transactionWrapper.apply(supplier, currentUser);
        }

        return (T) transactionWrapper.apply(supplier, null);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T byTenant(Supplier<T> supplier, SysUserDTO currentUser) {
        if (currentUser.getSysTenantVO() != null && currentUser.getSysTenantVO().getType() != TenantType.OPERATION) {
            return (T) transactionWrapper.apply(supplier, currentUser);
        }

        return (T) transactionWrapper.apply(supplier, null);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T byTenantAuto(Supplier<T> supplier) {
        var tenant = tenantClientProvider.getCurrentTenant();
        if (tenant != null && tenant.getType() != TenantType.OPERATION) {
            return (T) transactionWrapper.apply(supplier, tenant);
        }

        return (T) transactionWrapper.apply(supplier, null);
    }

    @Override
    public <T> void byAllTenant(Supplier<T> supplier) {
        var tenants = getAllTenant();
        for (SysTenantDTO tenant : tenants) {
            try {
                transactionWrapper.apply(supplier, tenant);
            } catch (Exception e) {
                log.error("处理租户【" + tenant.getId() + "】相关业务异常：", e);
            }
        }

        // 最后执行默认
        try {
            transactionWrapper.apply(supplier, null);
        } catch (Exception e) {
            log.error("处理默认租户相关业务异常：", e);
        }
    }

    /**
     * 获取所有租户信息
     *
     * @return 租户信息
     */
    private List<SysTenantDTO> getAllTenant() {
        return tenantClientProvider.getAllTenants();
    }

    /**
     * 根据租户ID获取租户信息
     *
     * @param sysTenantId 租户ID
     * @return 租户洗洗
     */
    private SysTenantDTO getTenant(Long sysTenantId) {
        return tenantClientProvider.getTenant(sysTenantId);
    }
}
