package com.elitescloud.boot.tenant.client.support.config;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.tenant.client.common.AbstractTenantIsolateDelegate;
import com.elitescloud.boot.wrapper.TransactionWrapper;
import com.elitescloud.cloudt.system.constant.TenantType;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import java.util.function.Supplier;

/**
 * 租户的事务处理.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/7
 */
@Log4j2
public class TenantTransactionWrapper extends AbstractTenantIsolateDelegate<Object, Object, TransactionStatus> implements TransactionWrapper<Object, Object> {

    private static final String TRANSACTION_NAME_PREFIX = "cloudt_tenant";

    private final PlatformTransactionManager transactionManager;

    public TenantTransactionWrapper(PlatformTransactionManager platformTransactionManager) {
        this.transactionManager = platformTransactionManager;
    }

    @Override
    public String transactionName() {
        return TRANSACTION_NAME_PREFIX;
    }

    @Override
    protected boolean supportApply(SysTenantDTO tenant) {
        String currentTransactionName = TransactionSynchronizationManager.getCurrentTransactionName();
        String newTransactionName = generateTransactionName(tenant);

        log.debug("Transaction try：{} -> {}", currentTransactionName, newTransactionName);
        return !CharSequenceUtil.equals(newTransactionName, currentTransactionName);
    }

    @Override
    protected TransactionStatus beforeApply(Supplier<Object> supplier, SysTenantDTO tenant) {
        String newTransactionName = generateTransactionName(tenant);

        log.debug("Transaction execute：{} -> {}", TransactionSynchronizationManager::getCurrentTransactionName, () -> newTransactionName);
        return buildTenantTransactionDefinition(newTransactionName);
    }

    @Override
    protected void afterCompletion(Object result, TransactionStatus transactionStatus, boolean useDefault, Exception exp) {
        if (transactionStatus == null) {
            return;
        }
        if (exp == null) {
            // 没有异常，则正常提交
            transactionManager.commit(transactionStatus);
            return;
        }

        //出现异常则回滚
        transactionManager.rollback(transactionStatus);
    }

    private TransactionStatus buildTenantTransactionDefinition(String newTransactionName) {
        DefaultTransactionDefinition definition = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        definition.setName(newTransactionName);
        return transactionManager.getTransaction(definition);
    }

    private String generateTransactionName(SysTenantDTO tenant) {
        return TRANSACTION_NAME_PREFIX + (tenant == null || tenant.getType() == TenantType.OPERATION ? "[default]" : "[" + tenant.getId() + "]");
    }
}
